/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.json.JSONException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.auth.AuthUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.GuardMimeHeaders;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.inputvalidation.RangeInputValidator;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgpcore.services.PGPMimeSigningService;
import com.openexchange.guard.pgpcore.services.PGPSignatureVerificationResultUtil;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;

/**
 * {@link VerifyMimeEmailAction}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.4
 */
public class VerifyMimeEmailAction extends GuardServletAction {

    private static final String MESSAGE_FIELD_NAME = "file";

    private static GuardParsedMimeMessage parseMimeMessage(HttpServletRequest request, InputStream message, int userId, int contextId) throws MessagingException, JSONException, OXException, FileUploadException, IOException {
        return ParseIncoming.parse(request, null, message, userId, contextId, null, null, false);
    }

    /**
     * Just sends a simple json data with true/false if signature verifies.
     * Returns data: missing if no signatures found
     * sendSimpleResponse
     *
     * @param response
     * @param verificationResults
     */
    private static void sendSimpleResponse(HttpServletResponse response, List<PGPSignatureVerificationResult> verificationResults) {
        if (verificationResults == null) {
            ServletUtils.sendObject(response, "missing");
            return;
        }
        boolean verified = false;
        for (PGPSignatureVerificationResult result : verificationResults) {
            verified = verified || result.isVerified();
        }
        ServletUtils.sendObject(response, verified);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        RangeInputValidator<Integer> positiveValidator = new RangeInputValidator<Integer>(0, Integer.MAX_VALUE);

        final boolean mandatory = true;
        final int userId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "user", mandatory), "user");
        final int contextId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "context", mandatory), "context");
        final boolean simpleResponse = ServletUtils.getBooleanParameter(request, "simple");

        FileUploadHandler fileUploadHandler = new FileUploadHandler(Services.getService(GuardConfigurationService.class));
        Collection<FileItem> items = fileUploadHandler.parseItems(request);
        try (InputStream messageStream = fileUploadHandler.getFileItemStreamFrom(items, MESSAGE_FIELD_NAME, true);) {
            final GuardParsedMimeMessage parsedMimeMessage = parseMimeMessage(request, messageStream, userId, contextId);
            UserIdentity userIdentity = AuthUtils.getUserIdentityFrom(userId, contextId);

            final MimeMessage messageToVerify = parsedMimeMessage.getMessage();

            //new:
            messageToVerify.removeHeader(GuardMimeHeaders.X_GUARD_SIGNATURE_RESULT);
            PGPMimeSigningService pgpMimeSigningService = Services.getService(PGPMimeSigningService.class);
            List<PGPSignatureVerificationResult> verificationResults = pgpMimeSigningService.verify(messageToVerify, userIdentity);
            if (simpleResponse) {
                sendSimpleResponse(response, verificationResults);
                return;
            }
            if (verificationResults != null) {
                String[] headers = PGPSignatureVerificationResultUtil.toHeaders(verificationResults);
                for (String header : headers) {
                    messageToVerify.addHeader(GuardMimeHeaders.X_GUARD_SIGNATURE_RESULT, header);
                }
            }
            messageToVerify.writeTo(response.getOutputStream());
        } catch (OXException ex) {
            if (simpleResponse && ex.similarTo(PGPCoreServicesExceptionCodes.SIGNATURE_ERROR_NO_SIGNED_DATA_IN_EMAIL)) {
                sendSimpleResponse(response, null);
                return;
            }
            throw ex;
        }
    }

}
