/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.OGPGPKeyRing;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseAction;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseWrapper;
import com.openexchange.guard.keymanagement.services.PublicExternalKeyService;
import com.openexchange.guard.mime.services.Attachment;
import com.openexchange.guard.mime.services.MimeEncryptionService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.ApiResponse;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.keymanagement.responses.ExternalPublicKeyRingCollectionResponse;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.keys.parsing.PGPPublicKeyRingFactory;

/**
 * Retrieves a public key sent as mail attachment
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GetPublicKeyFromAttachmentAction extends GuardServletAction {

    private static final boolean MANDATORY           = true;
    private static final String  PARAM_FOLDER        = "folder";
    private static final String  PARAM_EMAIL_ID      = "emailid";
    private static final String  PARAM_ATTACKMENT_ID = "attach";
    private static final String  PARAM_AUTH_TOKEN    = "auth";
    private static final String  PARAM_FILE_NAME     = "filename";
    private static final String  PARAM_INLINE        = "inline";

    private Api createOXApi(HttpServletRequest request) {
        return new Api(new OxCookie(request.getCookies()), request);
    }

    private PGPPublicKeyRing createKeyRing(String asciiKeyData) throws IOException {
        return PGPPublicKeyRingFactory.create(asciiKeyData);
    }

    private PGPPublicKeyRing createKeyRing(InputStream inputStream) throws IOException {
        return createKeyRing(new String(IOUtils.toByteArray(inputStream), StandardCharsets.UTF_8));
    }

    private PGPPublicKeyRing parseKeyRingData(Api oxapi, String folder, String emailid, String attachmentId) throws IOException {
        try(ApiResponse plainAttachment = oxapi.getPlainAttachment(emailid, attachmentId, folder)){
            Attachment att = new Attachment();
            att.setContent(plainAttachment.readContent());
            return createKeyRing(new String(att.getContent(), StandardCharsets.UTF_8));
        }
    }

    private PGPPublicKeyRing parsePGPInlineKeyData(UserIdentity userIdentity, Api oxapi, String folder, String emailid, String attachmentId) throws IllegalStateException, IOException, OXException {
        PGPCryptoService cryptoService = Services.getService(PGPCryptoService.class);
        try(ApiResponse plainAttachment = oxapi.getPlainAttachment(emailid, attachmentId, folder)){
            if (plainAttachment == null) {
                throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create("Unable to get attachment");
            }
            InputStream encryptedKeyData = plainAttachment.getContent();
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            cryptoService.decrypt(encryptedKeyData, output, userIdentity);
            return createKeyRing(new String(output.toByteArray(), StandardCharsets.UTF_8));
        }
    }

    private PGPPublicKeyRing parsePGPMimeKeyData(UserIdentity userIdentity, Api oxapi, String folder, String emailid, String filename) throws OXException, MessagingException, IOException {
        MimeMessage mimeMessage = new MimeMessage(
            Session.getDefaultInstance(new Properties()),
            new ByteArrayInputStream(oxapi.getMime(emailid, folder).getBytes(StandardCharsets.UTF_8)));
        MimeEncryptionService mimeCryptoService = Services.getService(MimeEncryptionService.class);
        Part decryptedPart = mimeCryptoService.doDecryption(mimeMessage, filename, userIdentity);
        return decryptedPart == null ? null : createKeyRing(decryptedPart.getInputStream());
    }

    private List<OGPGPKeyRing> importExternalPublicKeyring(GuardUserSession userSession, PGPPublicKeyRing parsedKeyRingData) throws OXException {
        PublicExternalKeyService externalKeyService = Services.getService(PublicExternalKeyService.class);
        return externalKeyService.importPublicKeyRing(userSession.getUserId(), userSession.getContextId(), parsedKeyRingData);
    }


    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response,GuardUserSession userSession) throws Exception {

        PGPPublicKeyRing parsedKeyRingData = null;

        String folder = ServletUtils.getStringParameter(request, PARAM_FOLDER,MANDATORY);
        String emailid = ServletUtils.getStringParameter(request, PARAM_EMAIL_ID, MANDATORY);
        String attachmentId = ServletUtils.getStringParameter(request, PARAM_ATTACKMENT_ID, MANDATORY);
        String auth = ServletUtils.getStringParameter(request, PARAM_AUTH_TOKEN, !MANDATORY);

        if(auth == null) {
            //The client wants to import the keyring from a plaintext email object
            parsedKeyRingData = parseKeyRingData(createOXApi(request), folder, emailid, attachmentId);
        } else {
            //The client wants to import the key from an encrypted email object
            UserIdentity userIdentity = requireUserIdentityFrom(userSession, auth);
            AntiAbuseWrapper antiAbuseWrapper = getAntiAbuseWrapper(
                AntiAbuseUtils.getAllowParameters(request, userIdentity.getIdentity(), new String(userIdentity.getPassword())));
            final String filename = ServletUtils.getStringParameter(request, PARAM_FILE_NAME, MANDATORY);
            final boolean inline = ServletUtils.getBooleanParameter(request,PARAM_INLINE,!MANDATORY);
            parsedKeyRingData = antiAbuseWrapper.doAction(new AntiAbuseAction<PGPPublicKeyRing>() {

                @Override
                public PGPPublicKeyRing doAction() throws Exception {
                    return  inline ?
                        parsePGPInlineKeyData(userIdentity, createOXApi(request), folder, emailid, attachmentId ) :
                        parsePGPMimeKeyData(userIdentity, createOXApi(request), folder, emailid, filename);
                }
            });

        }

        //Import the parsed key data as external key ring
        List<OGPGPKeyRing> importedKeys = importExternalPublicKeyring(userSession, parsedKeyRingData);
        ServletUtils.sendObject(response, new ExternalPublicKeyRingCollectionResponse(importedKeys));
    }
}
