/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.io.IOException;
import java.util.List;
import java.util.Objects;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.guard.keymanagement.commons.OGPGPKey;
import com.openexchange.guard.keymanagement.storage.OGPGPKeysStorage;
import com.openexchange.pgp.core.PGPKeyRetrievalStrategy;
import com.openexchange.pgp.keys.parsing.PGPPublicKeyRingFactory;

/**
 * {@link UserUploadedPGPKeyRetrievalStrategy}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.0
 */
public class UserUploadedPGPKeyRetrievalStrategy implements PGPKeyRetrievalStrategy {

    final private OGPGPKeysStorage keyStorage;
    private final int userId;
    private final int contextId;

    /**
     * Initializes a new {@link UserUploadedPGPKeyRetrievalStrategy}.
     *
     * @param userId The user id
     * @param contextId The context id of the user
     */
    public UserUploadedPGPKeyRetrievalStrategy(OGPGPKeysStorage keyStorage, int userId, int contextId) {
        this.keyStorage = Objects.requireNonNull(keyStorage, "keyStorage must not ne null");
        this.userId = userId;
        this.contextId = contextId;
    }

    /**
     * Gets the PGPPublicKey with the given id from the {@link OGPGPKey} object.
     *
     * @param ogpgpKey The object to extract the PGPPublicKey from.
     * @param keyId The ID of the public key to extract.
     * @return The PGPPublicKey with the given ID, or null if no such key was found.
     * @throws IOException
     */
    private PGPPublicKey getPublicKey(OGPGPKey ogpgpKey, long keyId) throws IOException {
        PGPPublicKeyRing keyRing = PGPPublicKeyRingFactory.create(ogpgpKey.getPublicPGPAscData());
        return keyRing.getPublicKey(keyId);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.pgp.core.PGPKeyRetrievalStrategy#getSecretKey(long, java.lang.String, char[])
     */
    @Override
    public PGPPrivateKey getSecretKey(long keyId, String userIdentity, char[] password) throws Exception {
        /* user does only upload public keys for external recipients */
        return null;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.pgp.core.PGPKeyRetrievalStrategy#getPublicKey(long)
     */
    @Override
    public PGPPublicKey getPublicKey(long keyId) throws Exception {
        List<OGPGPKey> publicKeys = keyStorage.getForUser(userId, contextId);
        for(OGPGPKey keyRing : publicKeys) {
            PGPPublicKey publicKey = getPublicKey(keyRing, keyId);
            if(publicKey != null) {
                return publicKey;
            }
        }
        return null;
    }
}
