package com.openexchange.guard.guestreset.storage.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guestreset.exceptions.GuardGuestResetExceptionCodes;
import com.openexchange.guard.guestreset.osgi.Services;
import com.openexchange.guard.guestreset.storage.TemporaryTokenService;
import com.openexchange.guard.guestreset.storage.Token;

public class TemporaryTokenServiceImpl implements TemporaryTokenService {

    @Override
    public void storeToken(Token token) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        Connection connection = guardDatabaseService.getWritableForGuard();
        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(TokenTableSql.INSERT_TOKEN);
            stmt.setString(1, token.getToken());
            stmt.setString(2, token.getEmail());
            stmt.setInt(3, token.getUserId());
            stmt.setInt(4, token.getContextId());
            stmt.setDate(5, new java.sql.Date(token.getCreated().getTime()));
            stmt.execute();
            return;
        } catch (SQLException e) {
            throw GuardGuestResetExceptionCodes.TOKEN_SAVE_ERROR.create(e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    @Override
    public Token getToken(String tokenString) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        GuardConfigurationService guardConfig = Services.getService(GuardConfigurationService.class);

        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(TokenTableSql.SELECT_TOKEN_FROM_ID);
            stmt.setString(1, tokenString);
            stmt.setInt(2, guardConfig.getIntProperty(GuardProperty.temporaryTokenLifespan));
            resultSet = stmt.executeQuery();
            if (resultSet.next()) {
                return createToken(resultSet);
            }
            return null;
        } catch (SQLException e) {
            throw GuardGuestResetExceptionCodes.TOKEN_RETIEVAL_ERROR.create(e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
    }

    private Token createToken(ResultSet rs) throws SQLException {
        String token = rs.getString("token");
        String email = rs.getString("email");
        int id = rs.getInt("id");
        int cid = rs.getInt("cid");
        Date created = rs.getDate("created");
        return new Token(token, email, id, cid, created);
    }

    @Override
    public void deleteToken(String tokenString) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        Connection connection = guardDatabaseService.getWritableForGuard();
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(TokenTableSql.DELETE_TOKEN);
            stmt.setString(1, tokenString);
            stmt.execute();
            return;
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }

    }

    @Override
    public void deleteToken(Token token) throws OXException {
        deleteToken(token.getToken());

    }

    @Override
    public void cleanupTokens(int hours) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        Connection connection = guardDatabaseService.getWritableForGuard();
        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(TokenTableSql.CLEANUP_TOKENS);
            stmt.setInt(1, hours);
            stmt.execute();
            return;
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);
        }

    }

}
