/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.file.internal;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.dataobject.EncryptedObject;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.storage.AbstractLegacyStorage;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.storage.file.osgi.Services;

/**
 * Class for saving encrypted object to and from file system.
 *
 * @author benjamin.otterbach@open-xchange.com
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class LegacyFileStorageImpl extends AbstractLegacyStorage {

    private static Logger logger = LoggerFactory.getLogger(File.class);

    /**
     * Initialises a new {@link LegacyFileStorageImpl}.
     *
     * @throws OXException
     */
    public LegacyFileStorageImpl(GuardConfigurationService guardConfigService) throws OXException {
        super();
        //GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        File file = new java.io.File(guardConfigService.getProperty(GuardProperty.uploadDirectory));
        if (!file.exists()) {
            if (!file.mkdir()) {
                throw GuardCoreExceptionCodes.IO_ERROR.create("Unable to create Guard upload Directory");
            };
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#writeObj(java.lang.String, java.lang.String, byte[])
     */
    @Override
    public String writeObj(String directoryPrefix, String ObjId, byte[] data) throws OXException {
        String result = null;
        OutputStream output = null;
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String directory = guardConfigService.getProperty(GuardProperty.uploadDirectory) + Storage.DELIMITER + directoryPrefix;

            File file = new File(directory);
            if (!file.exists()) {
                boolean created = file.mkdirs();
                if(!created) {
                    logger.error("Error creating directory " + file.toString());
                }
            }
            output = new BufferedOutputStream(new FileOutputStream(directory + Storage.DELIMITER + ObjId));
            output.write(data);
            result = directoryPrefix + Storage.DELIMITER + ObjId;
        } catch (Exception e) {
            logger.error("Error writing output stream for file " + ObjId, e);
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } finally {
            if (output != null) {
                try {
                    output.close();
                } catch (IOException e) {
                    throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
                }
            }
        }
        return result;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getEncrObj(java.lang.String, java.lang.String)
     */
    @Override
    public EncryptedObject getEncrObj(String directoryPrefix, String ObjId) {
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String directory = guardConfigService.getProperty(GuardProperty.uploadDirectory);
            InputStream in = new FileInputStream(directory + Storage.DELIMITER + directoryPrefix + Storage.DELIMITER + ObjId);
            EncryptedObject obj;
            try {
                obj = new EncryptedObject(in);
            } finally {
                if (in != null) {
                    in.close();
                }
            }
            return obj;
        } catch (FileNotFoundException e) {
            logger.error("File not found");
            EncryptedObject obj = new EncryptedObject();
            obj.setItemID("not found");
            return obj;
        } catch (Exception ex) {
            logger.error("Error getting encrypted file " + ObjId, ex);
            return null;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getObjectStream(java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getObjectStream(String directoryPrefix, String ObjId) {
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String directory = guardConfigService.getProperty(GuardProperty.uploadDirectory);
            InputStream in = new FileInputStream(directory + Storage.DELIMITER + directoryPrefix + Storage.DELIMITER + ObjId);
            return in;
        } catch (FileNotFoundException e) {
            logger.error("File not found");
            return null;
        } catch (Exception ex) {
            logger.error("Error getting encrypted file " + ObjId, ex);
            return null;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getObjectStream(java.lang.String)
     */
    @Override
    public InputStream getObjectStream(String location) {
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String directory = guardConfigService.getProperty(GuardProperty.uploadDirectory);
            InputStream in = new FileInputStream(directory + Storage.DELIMITER + location);
            return in;
        } catch (FileNotFoundException e) {
            logger.error("File not found");
            return null;
        } catch (Exception ex) {
            logger.error("Error getting encrypted file " + location, ex);
            return null;
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.Storage#objectExists(java.lang.String)
     */
    @Override
    public boolean objectExists(String location) throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        String directory = guardConfigService.getProperty(GuardProperty.uploadDirectory);
        String fullName = directory + Storage.DELIMITER + location;
        return new File(fullName).exists();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#deleteObj(java.lang.String)
     */
    @Override
    public void deleteObj(String location) throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        String dir = guardConfigService.getProperty(GuardProperty.uploadDirectory);
        String fileWithPath = dir + Storage.DELIMITER + location;

        File file = new File(fileWithPath);
        if (file.exists()) {
            if (!file.delete()) {
                logger.error("Error deleting file " + location);
            }
        }
        int i = fileWithPath.lastIndexOf(Storage.DELIMITER);
        File directory = new File(fileWithPath.substring(0, i));
        if (directory.isDirectory()) {
            if (directory.list().length == 0) {
                if (!directory.delete()) {
                    logger.error("Problem deleting directory " + directory.getPath());
                };
            }
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#deleteEncrObj(java.lang.String)
     */
    @Override
    public void deleteEncrObj(String location) throws OXException {
        deleteObj(location);
    }
}
