/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.fileupload;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.Objects;
import javax.mail.util.SharedFileInputStream;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItem;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.FileCleanerCleanup;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.output.DeferredFileOutputStream;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * {@link FileUploadHandler} is a simple wrapper around Apache Commons FileUpload.
 * <p>
 * It provides basic Apache Commons FileUpload functionality configured to be used within OX Guard.
 * </p>
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class FileUploadHandler {

    private static final String GUARD_FILE_SUFFIX = ".grd";
    private static final String GUARD_FILE_PREFIX = "guard-";
    private static final GuardCipherAlgorithm ENCRYPTION_ALGORITHM = GuardCipherAlgorithm.AES_CBC;

    private final GuardConfigurationService configurationService;
    private final boolean encrypted;

    /**
     * Initializes a new {@link FileUploadHandler}.
     *
     * @param configurationService The configuration to use
     */
    public FileUploadHandler(GuardConfigurationService configurationService) {
        this.configurationService = Objects.requireNonNull(configurationService, "configurationService must not be null");
        this.encrypted = false;
    }

    /**
     * Initializes a new {@link FileUploadHandler}.
     *
     * @param configurationService The configuration to use
     * @param encrypted True, in order to handle file uploads encrypted when buffered to disk, false to buffer as plain text
     */
    public FileUploadHandler(GuardConfigurationService configurationService, boolean encrypted) {
        this.configurationService = Objects.requireNonNull(configurationService, "configurationService must not be null");
        this.encrypted = encrypted;
    }

    /**
     * Internal method to get the directory in which the files should be stored temporary.
     *
     * @return The directory in which the files should be stored.
     */
    private File getUploadDirectory() {
        String directoryValue = configurationService.getProperty(GuardProperty.fileUploadDirectory);
        if (directoryValue != null && !directoryValue.trim().isEmpty()) {
            return new File(directoryValue);
        }
        return new File(System.getProperty("java.io.tmpdir"));
    }

    /**
     * Internal method to get the threshold value at which OX Guard begins to buffer uploaded files to disk before processing them.
     *
     * @return The threshold value
     */
    private int getThreshold() {
        return configurationService.getIntProperty(GuardProperty.fileUploadBufferThreshhold);
    }

    private long getConfiguredMaxUploadSize() {
        //Getting the maximum allowed upload size
        String maxSizeProperty = configurationService.getPropertyFromFile("server.properties", "MAX_UPLOAD_SIZE");
        long maxSize = -1L;
        if (maxSizeProperty != null) {
            maxSize = Long.parseLong(maxSizeProperty);
            if (maxSize == 0) {
                maxSize = -1;
            }
        }
        return maxSize;
    }

    /**
     * Creates the {@link ServletFileUpload}.
     *
     * @param request The request to create the {@link ServletFileUpload} for
     * @param maxUploadSize The max. allowed upload size
     * @return The configured, ready to use {@link ServletFileUpload}.
     * @throws OXException
     */
    private ServletFileUpload createFileUpload(HttpServletRequest request, long maxUploadSize) throws OXException {
        DiskFileItemFactory diskFileItemFactory = new DiskFileItemFactory(getThreshold(), getUploadDirectory());
        diskFileItemFactory.setFileCleaningTracker(FileCleanerCleanup.getFileCleaningTracker(request.getServletContext()));
        FileItemFactory fileItemFactory = diskFileItemFactory;
        if (encrypted) {
            try {
                fileItemFactory = new EncryptedFileItemFactory(fileItemFactory, ENCRYPTION_ALGORITHM);
            } catch (NoSuchAlgorithmException e) {
                throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
            }
        }
        ServletFileUpload fileUpload = new ServletFileUpload(fileItemFactory);
        fileUpload.setFileSizeMax(maxUploadSize);
        fileUpload.setSizeMax(maxUploadSize);
        return fileUpload;
    }

    /**
     * Creates a new {@link DeferredFileOutputStream},configured to be used with OX Guard, which can be used to buffer output before serving it to clients
     *
     * @return A new DeferredFileOutputStream
     */
    public DeferredFileOutputStream createOutputStream() {
        return new DeferredFileOutputStream(getThreshold(), GUARD_FILE_PREFIX, GUARD_FILE_SUFFIX, getUploadDirectory());
    }

    /**
     * Parses all {@link FileItem} instances from a given request with the default configured max. allowed upload size
     *
     * @param request The request
     * @return A collection of parsed FileItems.
     * @throws FileUploadException
     */
    public Collection<FileItem> parseItems(HttpServletRequest request) throws OXException {
        try {
            return createFileUpload(request, getConfiguredMaxUploadSize()).parseRequest(request);
        } catch (FileUploadException e) {
            throw GuardCoreExceptionCodes.MULTIPART_UPLOAD_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Parses all {@link FileItem} instances from a given request with a max. upload size
     *
     * @param request The request
     * @param maxUploadSize The max. allowed upload size
     * @return A collection of parsed FileItems.
     * @throws FileUploadException
     */
    public Collection<FileItem> parseItems(HttpServletRequest request,long maxUploadSize) throws OXException {
        try {
            return createFileUpload(request, maxUploadSize).parseRequest(request);
        } catch (FileUploadException e) {
            throw GuardCoreExceptionCodes.MULTIPART_UPLOAD_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Parses a {@link FileItemIterator} from the given request with the default configured max. allowed upload size
     *
     * @param request The request
     * @return The {@link FileItemIterator} parsed from the given request.
     * @throws FileUploadException
     * @throws IOException
     */
    public FileItemIterator parseIterator(HttpServletRequest request) throws OXException {
        try {
            return createFileUpload(request, getConfiguredMaxUploadSize()).getItemIterator(request);
        } catch (FileUploadException e) {
            throw GuardCoreExceptionCodes.MULTIPART_UPLOAD_ERROR.create(e, e.getMessage());
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }


    /**
     * Gets a specific InputStream for a {@link FileItem} from a parsed collection of fileItems.
     *
     * @param fileItems The parsed collection of FileItems
     * @param name The name of the item to get the stream for
     * @param Whether the FileItem is mandatory or not
     * @return The InputStream to the FileItem with the given name, or null if no such FileItem was found and mandatory is set to false.
     * @throws IOException
     * @throws OXException If mandatory is set to true and the given FileItem was not found.
     */
    public InputStream getFileItemStreamFrom(Collection<FileItem> fileItems, String name, boolean mandatory) throws OXException {
        try {
            for (FileItem fileItem : fileItems) {
                if (fileItem.getFieldName().equals(name)) {
                    if(!fileItem.isInMemory()) {
                       File tmpFile = null;
                       if(fileItem instanceof DiskFileItem) {
                           tmpFile = ((DiskFileItem)fileItem).getStoreLocation();
                           return new SharedFileInputStream(tmpFile);
                       }
                       else if(fileItem instanceof EncryptedFileItem) {
                           EncryptedFileItem encryptedFileItem = (EncryptedFileItem)fileItem;
                           return encryptedFileItem.getSharedInputStream();
                       }
                    }
                    return fileItem.getInputStream();
                }
            }
            if (mandatory) {
                throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(name);
            }
            return null;
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }
}
