/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.wksclient;

import java.util.Iterator;
import java.util.Objects;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.guard.keymanagement.commons.trust.KeySource;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;

/**
 * {@link WKSResult}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.2
 */
public class WKSResult {

    private final PGPPublicKeyRing publicKeyRing;
    private KeySource keySource;

    /**
     * Initializes a new {@link WKSResult}.
     *
     * @param publicKeyRing The public key ring
     */
    public WKSResult(PGPPublicKeyRing publicKeyRing, KeySource keySource) {
        this.publicKeyRing = Objects.requireNonNull(publicKeyRing, "publicKeyRing must not be null");
        this.keySource = Objects.requireNonNull(keySource, "keySource must not be null");
    }

    /**
     * Gets the publicKeyRing
     *
     * @return The publicKeyRing
     */
    public PGPPublicKeyRing getPublicKeyRing() {
        return publicKeyRing;
    }

    /**
     * Returns whether or not the represented PGP key ring is expired.
     *
     * @return true if the represented PGP key is expired, false otherwise
     */
    public boolean isExpired() {
        Iterator<PGPPublicKey> iter = publicKeyRing.getPublicKeys();
        while (iter.hasNext()) {
            if (!PGPKeysUtil.isExpired(iter.next())) {
                return false;  // Check that we have at least one key not expired
            }
        }
        return true;
    }

    /**
     * Returns whether or not the represented PGP key ring is revoked.
     *
     * @return true if the represented PGP key is revoked, false otherwise
     */
    public boolean isRevoked() {
        Iterator<PGPPublicKey> iter = publicKeyRing.getPublicKeys();
        while (iter.hasNext()) {
            if (!iter.next().hasRevocation()) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks that we have at least one valid encryption key
     *
     * @return true if a valid encryption key found
     */
    public boolean hasValidKey() {
        Iterator<PGPPublicKey> iter = publicKeyRing.getPublicKeys();
        while (iter.hasNext()) {
            PGPPublicKey key = iter.next();
            if (!key.hasRevocation() && !PGPKeysUtil.isExpired(key) && key.isEncryptionKey()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the source of the result
     *
     * @return The source of the result
     */
    public KeySource getKeySource() {
        return keySource;
    }

    /**
     * Sets the source of the result.
     *
     * @param source The source of the result
     */
    public void setKeySource(KeySource source) {
        this.keySource = source;
    }
}