/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.translation;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * {@link ConditionalVissibleTemplateTransformation} allows to define conditional elements in templates.
 *
 * <p>Put the following pattern in your template in order define conditional content: #if(name)..#endif
 *
 * <p>If the condition passed to the constructor is true, {@link ConditionalVissibleTemplateTransformation} will remove the outer parts of the pattern, which are #if(name) and #endif, while the inner text will be remaining.
 * If the condition is false, the whole pattern including the inner text will be removed.
 */
public class ConditionalVissibleTemplateTransformation implements TemplateTransformation {

    private final String name;
    boolean transform;

    /**
     *
     * Initializes a new {@link ConditionalVissibleTemplateTransformation}.
     * 
     * @param name The name of the condition
     * @param condition true, if the inner text of the condition pattern should remain, false, if the whole pattern should be removed
     */
    public ConditionalVissibleTemplateTransformation(String name, boolean condition) {
        this.name = name;
        this.transform = condition;
    }

    @Override
    public String transformTemplate(String template) {
        String regex = "#if\\((" + name + ")\\)(.*?)(#endif)";
        Pattern pattern = Pattern.compile(regex, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(template);
        while (matcher.find()) {
            if (matcher.group(1).equals(name)) {
                if (transform) {
                    //removing everything
                    template = matcher.replaceAll("");
                } else {
                    //Only keeping the middle group (without the #if and #endif groups)
                    template = matcher.replaceAll("$2");
                }
            }
        }
        return template;
    }
}
