/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.responses;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.io.IOUtils;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.ContentTypeDetectionUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * {@link BinaryResponse} serves binary content to a client.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class BinaryResponse {

    private final static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(BinaryResponse.class);
    private final HttpServletResponse response;
    private static final List<String> INLINE_WHITE_LIST;
    private static final String APPLICATION_OCTETSTREAM = "application/octet-stream";

    static {
        //A list of content-types for which we allow inline content-disposition
        INLINE_WHITE_LIST = Arrays.asList(new String[] {
                "image/jpeg",
                "image/png",
                "image/gif",
        });
    }

    /**
     * Initializes a new {@link BinaryResponse}.
     *
     * @param response The response to use
     */
    public BinaryResponse(HttpServletResponse response) {
        this.response = response;
    }

    /**
     * Internal method to check if we send the binary data as disposition "attachment" for security reason.
     *
     * @param contentType The content type of the binary data
     * @return true, if the binary data must be served as disposition "attachment", false otherwise
     * @throws OXException
     */
    private boolean forceServeAsDownload(String contentType) throws OXException {
        if (contentType != null) {
            return !INLINE_WHITE_LIST.contains(contentType);
        }
        return true;
    }

    /**
     * Internal method to guess the content type of the binary data
     *
     * @param data The data to guess the content-type for
     * @return The guessed content-type, or "application/octet-stream" if the content-type could not be specified.
     * @throws OXException
     */
    private String guessContentType(InputStream data) {
        try {
            final String contentType = ContentTypeDetectionUtil.guessContentType(data);
            if (contentType != null) {
                return contentType;
            }
            else{
                logger.debug("Unknown content type for binary data");
            }
        } catch (final Exception e) {
            logger.error("Error while detecting content-type for binary data: {} ", e.getMessage());
        }
        return APPLICATION_OCTETSTREAM;
    }

    /**
     * Serves binary data
     *
     * @param data The binary data
     * @param serveAsDownload True to force content-disposition being set to "attachment", false for auto-detection
     * @param fileName The filename associated with the data
     * @throws OXException
     */
    public void send(InputStream data, boolean serveAsDownload, String fileName) throws OXException {
        try {
            data = Objects.requireNonNull(data, "data must not be null");
            data = data.markSupported() ? data : new BufferedInputStream(data, ContentTypeDetectionUtil.READ_SIZE);
            final String contentType = guessContentType(data);

            //Setting content-disposition to "attachment" if desired by the caller, or if forced
            if (serveAsDownload || forceServeAsDownload(contentType)) {
                fileName = Objects.requireNonNull(fileName, "fileName must not be null");
                response.setHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");
            }

            //Setting the guessed contentType
            response.setContentType(contentType);
            //instruct the client to respect our contentType and not to sniff it in order to prevent
            //XSS if content-disposition is not 'attachment'
            response.setHeader("X-Content-Type-Options", "nosniff");

            //serve the data
            IOUtils.copy(data, response.getOutputStream());
        } catch (final IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

}
