/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.auth;

import java.util.Collection;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProductName;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guest.redirector.GuestRedirectorService;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.logging.GuardLogProperties;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.user.GuardCapabilities;

/**
 * This class is largely depreciated.  Used only for old Guest email links
 * The old Guest reader is not needed for installations that did not have users < 2.10
 * This is only used to log in to the Guest reader for upgrade/redirect of old links
 *
 * {@link GuestLoginHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class GuestLoginHandler {

    private static Logger LOG = LoggerFactory.getLogger(GuestLoginHandler.class);
    private final static String ENCR_PASSWORD_AUTH_FIELD_NAME = "encr_password";

    /**
     * Authorize login from guest account
     * Only used for old Guest reader.  Not used for new users
     *
     * @param request
     * @param response
     * @throws OXException
     */
    public void guestlogin(HttpServletRequest request, HttpServletResponse response) throws OXException {
        JsonObject login_json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        response.setContentType("application/json");
        // response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String sessionID = CipherUtil.getUUID();
        String cookieID = CipherUtil.getUUID();
        Cookie cookie = new Cookie("OxReaderID", cookieID);
        //cookie.setHttpOnly(true);
        cookie.setPath("/");
        response.addCookie(cookie);
        String username = JsonUtil.getStringFromJson(login_json, "username", true);
        if(username.isEmpty()) {
            throw GuardCoreExceptionCodes.JSON_PARAMETER_MISSING.create(username);
        }
        String pin = login_json.get("pin") == null ? null : login_json.get("pin").getAsString();
        try {

            String password = JsonUtil.getStringFromJson(login_json, "password", true);
            GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
            GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
            if (antiabuse.blockLogin(AntiAbuseUtils.getAllowParameters(request, username, password))) {
                response.getWriter().write("{\"auth\":\"lockout\", \"time:\": " + configService.getIntProperty(GuardProperty.badMinuteLock) + "}");
                LOG.info("Guest auth lockout for " + username);
                return;
            }


            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys keys = keyService.getKeys(username);
            if (keys == null) {
                response.getWriter().write("{\"auth\":\"No Key\"}");
                LOG.info("Guest auth No Key for " + username);
                return;
            }
            GuardSessionService sessionService = Services.getService(GuardSessionService.class);
            String token = sessionService.newToken(cookieID, keys.getUserid(), keys.getContextid());
            if (token == null) {
                response.getWriter().write("{\"auth\":\"Unable to get token\"}");
                LOG.error("Guest auth Unable to get database token for " + username);
                return;
            }
            // This is the json we're going to encrypt and include in the auth
            JsonObject json = new JsonObject();
            json.addProperty("user_id", keys.getUserid());
            json.addProperty("encr_password", password);
            json.addProperty("sessionID", sessionID);
            json.addProperty("cid", keys.getContextid());
            json.addProperty("recoveryAvail", keys.isRecoveryAvail());

            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_GCM);
            String returndata = "Bad Password";

            // If Guest, and no password provided, possibly first time user
            if (password.equals("") && keys.getContextid() < 0) {
                // Only works if recoveryAvail and configured to allow no password on first use
                if (keys.isRecoveryAvail() && !configService.getBooleanProperty(GuardProperty.newGuestsRequirePassword, keys.getUserid(), keys.getContextid())) {
                    if (keys.getLastup() == null) {  // Make sure password never changed
                        JsonObject sendjson = new JsonObject();
                        sendjson.addProperty("productName", GuardProductName.getProductName((request.getServerName())));
                        // If PIN
                        if ((keys.getLastup() == null) && (keys.getQuestion() == null ? false : keys.getQuestion().equals("PIN"))) {
                              sendjson.addProperty("pin", true);
                            }
                        sendjson.addProperty("auth", "new");
                        //Getting generic guest settings from configuration
                        GuardCapabilities settings = getGenericGuestPermissions(keys.getContextid());
                        //converting settings to JSON
                        sendjson.add("settings", AuthUtils.getSettings(keys.getSettings().toString(), settings, keys.getUserid(), keys.getContextid()));
                        ServletUtils.sendJsonOK(response, sendjson);
                        return;
                    }
                }
            }

            //Check password
            boolean passwordVerified = false;
            if (!PGPUtil.verifyPassword(keys.getPGPSecretKeyRing(),password, keys.getSalt())) {
                //Check the other keys according to Bug 49675
                Collection<GuardKeys> otherKeys = keyService.getAllKeys(keys.getUserid(), keys.getContextid());
                for(GuardKeys otherKey : otherKeys) {
                    if(otherKey.getKeyid() == keys.getKeyid()) {
                        otherKeys.remove(otherKey);
                        break;
                    }
                }
                for(GuardKeys otherKey : otherKeys) {
                    if(PGPUtil.verifyPassword(otherKey.getPGPSecretKeyRing(),password, otherKey.getSalt())) {
                        passwordVerified = true;
                        keys = otherKey;
                        break;
                    }
                }
            }
            else {
                passwordVerified = true;
            }

            if (!passwordVerified) {
                antiabuse.report(AntiAbuseUtils.getReportParameter(false, request, username, password));
                LOG.info("Bad password for " + username);
                JsonObject sendjson = new JsonObject();
                sendjson.addProperty("auth", returndata);
                sendjson.addProperty("recoveryAvail", keys.isRecoveryAvail());
                ServletUtils.sendJsonOK(response, sendjson);
                return;
            } else {
                // Check if PIN
                if ((keys.getLastup() == null) && (keys.getQuestion() == null ? false : keys.getQuestion().equals("PIN"))) {
                    if (pin != null) {
                        if (pin.equals(keys.getAnswer())) {
                            returndata = cipherService.encrypt(
                                    AuthUtils.createAuthJson(
                                        keys.getUserid(),
                                        keys.getContextid(),
                                        keys.getEmail(),
                                        ENCR_PASSWORD_AUTH_FIELD_NAME,
                                        json.has("encr_password") ? json.get("encr_password").getAsString() : "",
                                        json.has("language") ? json.get("language").getAsString() : "").toString(),
                                    token);
                        } else {
                            returndata = "Bad";
                            antiabuse.addBad(username);
                        }
                    } else {
                        LOG.info("New Guest login, pin required");
                        returndata = "PIN";
                    }
                } else {
                    antiabuse.report(AntiAbuseUtils.getReportParameter(true, request, username, password));
                    returndata = cipherService.encrypt(
                            AuthUtils.createAuthJson(
                                keys.getUserid(),
                                keys.getContextid(),
                                keys.getEmail(),
                                ENCR_PASSWORD_AUTH_FIELD_NAME,
                                json.has("encr_password") ? json.get("encr_password").getAsString() : "",
                                json.has("language") ? json.get("language").getAsString() : "").toString(),
                            token);
                }
            }

            // The json returned in plain text.  no password here (encrypted within auth)  Include settings
            JsonObject sendjson = new JsonObject();
            sendjson.addProperty("productName", GuardProductName.getProductName((request.getServerName())));
            sendjson.addProperty("auth", returndata);
            sendjson.addProperty("userid", keys.getUserid());
            sendjson.addProperty("sessionID", sessionID);
            sendjson.addProperty("cid", keys.getContextid());
            sendjson.addProperty("recoveryAvail", keys.isRecoveryAvail());
            sendjson.addProperty("lastMod", keys.getLastup() == null ? "null" : keys.getLastup().toString());
            sendjson.addProperty("new", keys.getQuestion() == null ? true : false);
            // Since upgrade to 7.10, check for redirect
            GuestRedirectorService redirector = Services.getService(GuestRedirectorService.class);
            String redirect = redirector.getRedirect(keys.getUserid(), keys.getContextid());
            if (redirect != null) {
                sendjson.addProperty("redirect", redirect);
            }
            try {
                //Getting generic guest settings from configuration
                GuardCapabilities settings = getGenericGuestPermissions(keys.getContextid());
                //converting settings to JSON
                sendjson.add("settings", AuthUtils.getSettings(keys.getSettings().toString(), settings, keys.getUserid(), keys.getContextid()));
            } catch (Exception ex) {
                LOG.error("problem getting guest settings ", ex);
            }

            GuardLogProperties.put(GuardLogProperties.Name.GUEST_USER_ID, keys.getUserid());
            GuardLogProperties.put(GuardLogProperties.Name.GUEST_CONTEXT_ID, keys.getContextid());
            GuardLogProperties.put(GuardLogProperties.Name.GUEST_MAIL, username);
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_AUTH_ID, returndata);

            LOG.info("Guest auth good for " + username);
            ServletUtils.sendJsonOK(response, sendjson);

        } catch (Exception e) {
            LOG.error("Error during guest login  for user " + username, e);
        }
    }

    public GuardCapabilities getGenericGuestPermissions(int cid) throws OXException {
        GuardCapabilities guestSettings = new GuardCapabilities();
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);

        guestSettings.setNoDeleteRecovery(configService.getBooleanProperty(GuardProperty.noDeleteRecovery, -1, cid));
        guestSettings.setNorecovery(configService.getBooleanProperty(GuardProperty.noRecovery, -1, cid));
        guestSettings.setNoDeletePrivate(configService.getBooleanProperty(GuardProperty.noDeletePrivate, -1, cid));

        return guestSettings;
    }

}
