/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.wksclient.impl;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.Name;
import org.xbill.DNS.SRVRecord;
import org.xbill.DNS.TextParseException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.dns.DNSResult;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.dns.ValidationResult;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.inputvalidation.DomainNameValidator;
import com.openexchange.guard.wksclient.WKSClientService;
import com.openexchange.guard.wksclient.WKSResult;

/**
 * {@link SRVWKSClientService} - WKS client implementation which tries to obtain host and port from DNS SRV records, if not specified otherwise.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.2
 */
public class SRVWKSClientService implements WKSClientService {

    private static final Logger logger = LoggerFactory.getLogger(SRVWKSClientService.class);

    private final WKSClientService delegate;
    private final DNSService dnsService;
    private static final String WKS_SRV_SERVICE_NAME = "_openpgpkey._tcp.";

    private class WKSHost {

       private String host;
       private int port;
       private ValidationResult dnssecValidationResult;

       WKSHost(String host, int port, ValidationResult dnssecValidated){
          this.host = host;
          this.port = port;
          this.dnssecValidationResult = dnssecValidated;
       }

       boolean isValidated() {
           return dnssecValidationResult != null && dnssecValidationResult.isValidated();
       }
    }

    /**
     * Initializes a new {@link SRVWKSClientService}.
     *
     * @param delegate The actual {@link WKSClientService} to use
     * @param dnsService The {@link DNSService} to use
     */
    public SRVWKSClientService(WKSClientService delegate, DNSService dnsService) {
        this.delegate = Objects.requireNonNull(delegate, "delegate must not be null.");
        this.dnsService = Objects.requireNonNull(dnsService, "dnsService must not be null.");
    }

    private String getDomainFromEmail(String email) {
        String host =  email.substring(email.indexOf("@") + 1);
        if(!host.endsWith(".")) {
           host += ".";
        }
        return host;
    }

    private WKSHost toWKSHost(String originalDomain, SRVRecord record, ValidationResult dnssecValition) throws OXException {
        if(record != null) {
           try {
               Name target = record.getTarget();
               //We need to verify that the alternate hostname is...
               //a) ...a real domain name (and not something malicious like "attacker.com/wkd-test?.example.com.").
               String hostName = new DomainNameValidator().getInput(target.toString());
               if(hostName != null) {
                   //b)...a subdomain of the original host (required from the WKS RFC)
                   if(target.subdomain(new Name(originalDomain))) {
                       return new WKSHost(record.getTarget().toString(), record.getPort(), dnssecValition);
                   }
                   else {
                       logger.info("Discarding WKS SRV record {}, because {} must be a sub-domain of {}.",  target.toString(), target.toString(), originalDomain);
                   }
               }
               else {
                   logger.info("Discarding WKS SRV record {}, because it does not seem to be valid domain.", target.toString());
               }
           }
           catch(TextParseException e) {
               throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
           }
        }
        return null;
    }

    private WKSHost getHostFor(String email) throws OXException{
        DNSResult result = dnsService.getSrvs(WKS_SRV_SERVICE_NAME, email);
        if(result != null) {
            List<SRVRecord> records = result.getResultRecords();
            if(records != null && !records.isEmpty()) {
                SRVRecord record = records.get(0);
                return toWKSHost(getDomainFromEmail(email), record, result.getValidationResult());
            }
        }
        return null;
    }

    private Optional<WKSResult> updateKeySource(Optional<WKSResult> result, WKSHost alternateHost){
        if(alternateHost != null && alternateHost.isValidated()) {
            if(result != null && result.isPresent()) {
                result.get().setKeySource(WKSKeySources.PUBLIC_WKS_SRV_DNSSEC);
            }
        }
        return result;
    }

    private Collection<WKSResult> updateKeySource(Collection<WKSResult> results, WKSHost alternateHost){
        if(alternateHost != null && alternateHost.isValidated()) {
            if(results != null && !results.isEmpty()) {
                results.forEach(r -> r.setKeySource(WKSKeySources.PUBLIC_WKS_SRV_DNSSEC));
            }
        }
        return results;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.wksclient.WKSClientService#findAll(java.lang.String, int)
     */
    @Override
    public Collection<WKSResult> findAll(String email, int timeout) throws OXException {
        WKSHost alternateHost = getHostFor(email);
        return alternateHost != null ?
            updateKeySource(delegate.findAll(email, timeout, alternateHost.host, alternateHost.port), alternateHost) :
            delegate.findAll(email, timeout);
    }


    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.wksclient.WKSClientService#findAll(java.lang.String, int, java.lang.String, int)
     */
    @Override
    public Collection<WKSResult> findAll(String email, int timeout, String host, int port) throws OXException {
        //We do use the real host and port as requested by the caller
        return delegate.findAll(email, timeout, host, port);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.wksclient.WKSClientService#find(java.lang.String, int)
     */
    @Override
    public Optional<WKSResult> find(String email, int timeout) throws OXException {
        WKSHost alternateHost = getHostFor(email);
        return alternateHost != null ?
            updateKeySource(delegate.find(email, timeout, alternateHost.host, alternateHost.port), alternateHost) :
            delegate.find(email, timeout);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.wksclient.WKSClientService#find(java.lang.String, int, java.lang.String, int)
     */
    @Override
    public Optional<WKSResult> find(String email, int timeout, String host, int port) throws OXException {
        //We do use the real host and port as requested by the caller
        return delegate.find(email, timeout, host, port);
    }
}
