/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.secondfactor.impl;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.guard.database.utils.GuardConnectionWrapper;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.secondfactor.SecondFactorService;
import com.openexchange.guard.secondfactor.exceptions.PinExceptionCodes;
import com.openexchange.guard.secondfactor.osgi.Services;

/**
 * {@link PinImplementation}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class PinImplementation implements SecondFactorService {

    private String getPin (int userid, int cid) throws OXException {
        // only Guests have pin currently
        if (cid >= 0) return null;

        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getReadOnly(userid, cid, 0);
        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connectionWrapper.getConnection().prepareStatement(PinImplSql.GET_PIN);
            stmt.setInt(1, userid);
            stmt.setInt(2, cid);

            resultSet = stmt.executeQuery();
            if (resultSet.next()) {
                return resultSet.getString(1);
            }
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e,e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnly(connectionWrapper);
        }
        return null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.secondfactor.SecondFactorService#hasSecondFactor(int, int)
     */
    @Override
    public boolean hasSecondFactor(int userid, int cid) throws OXException {
        return (getPin(userid, cid) != null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.secondfactor.SecondFactorService#addSecondFactor(int, int, java.lang.String)
     */
    @Override
    public boolean addSecondFactor(int userid, int cid, String token) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getWritable(userid, cid, 0);
        PreparedStatement stmt = null;
        try {
            stmt = connectionWrapper.getConnection().prepareStatement(PinImplSql.SET_PIN);
            stmt.setInt(1, userid);
            stmt.setInt(2, cid);
            stmt.setString(3, token);

            return stmt.execute();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e,e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backReadOnly(connectionWrapper);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.secondfactor.SecondFactorService#verifySecondFactor(int, int, java.lang.String)
     */
    @Override
    public void verifySecondFactor(int userid, int cid, String token) throws OXException {
        if (!token.equals(getPin(userid, cid))) {
            throw PinExceptionCodes.BAD_PIN.create();

        }

    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.secondfactor.SecondFactorService#removeSecondFactor(int, int)
     */
    @Override
    public void removeSecondFactor(int userid, int cid) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);

        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getWritable(userid, cid, 0);
        PreparedStatement stmt = null;
        try {
            stmt = connectionWrapper.getConnection().prepareStatement(PinImplSql.DELETE_PIN);
            stmt.setInt(1, userid);
            stmt.setInt(2, cid);
            stmt.execute();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e,e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backReadOnly(connectionWrapper);
        }

    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.secondfactor.SecondFactorService#removeSecondFactorIfSingleUse(int, int)
     */
    @Override
    public void removeSecondFactorIfSingleUse(int userid, int cid) throws OXException {
        // pin is currently always single use
        removeSecondFactor(userid, cid);

    }

}
