/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl.osgi;

import java.security.Security;
import java.util.ArrayList;
import java.util.Collection;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.osgi.framework.BundleContext;
import com.openexchange.contact.ContactService;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.autocrypt.AutoCryptService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.hkpclient.services.HKPClientService;
import com.openexchange.guard.keymanagement.keysources.CompositeKeyPairSource;
import com.openexchange.guard.keymanagement.keysources.DbCachingKeyPairSource;
import com.openexchange.guard.keymanagement.keysources.RealtimeKeyPairSource;
import com.openexchange.guard.keymanagement.services.AccountCreationService;
import com.openexchange.guard.keymanagement.services.ContactKeyService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.services.ImportSystemPublicKey;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.KeyImportService;
import com.openexchange.guard.keymanagement.services.KeyPairSource;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.keymanagement.services.PublicExternalKeyService;
import com.openexchange.guard.keymanagement.services.RecipKeyService;
import com.openexchange.guard.keymanagement.services.impl.AccountCreationServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.ContactKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.GuardKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.GuardMasterKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.ImportSystemPublicKeyImpl;
import com.openexchange.guard.keymanagement.services.impl.KeyCreationServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.KeyImportServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.KeyRecoveryServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.PasswordChangeServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.PublicExternalKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.RecipKeyServiceImpl;
import com.openexchange.guard.keymanagement.storage.KeyCacheStorage;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.keymanagement.storage.OGPGPKeysStorage;
import com.openexchange.guard.keymanagement.storage.PGPKeysStorage;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.secondfactor.SecondFactorService;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.guard.wksclient.WKSClientService;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.pgp.keys.parsing.PGPKeyRingParser;

/**
 * {@link KeyManagementServicesImplActivator}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyManagementServicesImplActivator extends HousekeepingActivator {

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { GuardCipherFactoryService.class, GuardConfigurationService.class, GuardShardingService.class, GuardRatifierService.class,
                                KeyTableStorage.class, EmailStorage.class, KeyCacheStorage.class, PGPKeysStorage.class, OGPGPKeysStorage.class,
                                OXUserService.class, GuardAntiAbuseService.class, GuardSessionService.class, GuardTranslationService.class,
                                HKPClientService.class, WKSClientService.class, PGPKeyRingParser.class, ContactService.class, SecondFactorService.class};
    }

    @Override
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] { AutoCryptService.class };
    }

    private int getSymetricKeyAlgorithm() {
        GuardConfigurationService guardConfigurationService = getService(GuardConfigurationService.class);
        int rsaKeyLength = guardConfigurationService.getIntProperty(GuardProperty.aesKeyLength);
        switch(rsaKeyLength) {
            case 128:
                return PGPEncryptedData.AES_128;
            case 192:
                return PGPEncryptedData.AES_192;
            case 256:
                return PGPEncryptedData.AES_256;
            default:
                throw new IllegalArgumentException("Invalid configured AES keysize: " + rsaKeyLength);
        }
    }

    private int getKeyExpirationDays() {
        GuardConfigurationService guardConfigurationService = getService(GuardConfigurationService.class);
        return guardConfigurationService.getIntProperty(GuardProperty.keyValidDays);
    }

    private KeyPairSource[] getKeyPairCreationStrategies(GuardConfigurationService guardConfigurationService,
                                                         GuardCipherService guardCipherService,
                                                         GuardMasterKeyService guardMasterKeyService,
                                                         KeyCacheStorage keyCacheStorage){
        Collection<KeyPairSource >strategies = new ArrayList<KeyPairSource>();
        if(guardConfigurationService.getBooleanProperty(GuardProperty.rsaCache)) {
            //Using pre generated Key data from the key cache
            strategies.add(new DbCachingKeyPairSource(guardCipherService, guardMasterKeyService, keyCacheStorage));
        }
        strategies.add(new RealtimeKeyPairSource(guardConfigurationService));
        return strategies.toArray(new KeyPairSource[strategies.size()]);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        org.slf4j.LoggerFactory.getLogger(KeyManagementServicesImplActivator.class).info("Starting bundle: {}", context.getBundle().getSymbolicName());

        Services.setServiceLookup(this);
        Security.addProvider(new BouncyCastleProvider());

        final GuardConfigurationService guardConfigurationService = getService(GuardConfigurationService.class);
        final GuardCipherService guardCipherService = getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        final KeyCacheStorage keyCacheStorage = getService(KeyCacheStorage.class);

        // Register key services
        GuardKeyService keyService = new GuardKeyServiceImpl();
        registerService(GuardKeyService.class, keyService);
        trackService(GuardKeyService.class);

        //register the key recovery service
        KeyRecoveryService keyRecoveryService = new KeyRecoveryServiceImpl(getService(GuardConfigurationService.class), getService(OXUserService.class));
        registerService(KeyRecoveryService.class, keyRecoveryService);
        trackService(KeyRecoveryService.class);

        //create and registering the Key creation service
        KeyCreationService keyCreationService = new KeyCreationServiceImpl(guardCipherService,
                                                                           keyService,
                                                                           keyRecoveryService,
                                                                           getKeyExpirationDays(),
                                                                           getSymetricKeyAlgorithm());
        registerService(KeyCreationService.class, keyCreationService);
        trackService(KeyCreationService.class);

        // Password change service
        PasswordChangeService passService = new PasswordChangeServiceImpl();
        registerService(PasswordChangeService.class, passService);
        trackService(PasswordChangeService.class);

        //registering the master key Service
        GuardMasterKeyService guardMasterKeyService = new GuardMasterKeyServiceImpl(guardConfigurationService, keyService, keyCreationService);
        KeyPairSource keyPairSource = new CompositeKeyPairSource(getKeyPairCreationStrategies(guardConfigurationService, guardCipherService, guardMasterKeyService, keyCacheStorage));
        keyCreationService.setKeyPairSource(keyPairSource);
        guardMasterKeyService.initialise();
        registerService(GuardMasterKeyService.class, guardMasterKeyService );
        trackService(GuardMasterKeyService.class);

        //Register the RecipKey lookup service
        registerService(RecipKeyService.class, new RecipKeyServiceImpl(keyService,
            getService(GuardConfigurationService.class).getIntProperty(GuardProperty.remoteKeyLookupTimeout)));
        trackService(RecipKeyService.class);

        //Service for external public key handling
        PublicExternalKeyService publicExternalKeyService = new PublicExternalKeyServiceImpl(getService(GuardRatifierService.class), getService(OGPGPKeysStorage.class));
        registerService(PublicExternalKeyService.class, publicExternalKeyService);

        //Service for getting keys per contact
        registerService(ContactKeyService.class,
            new ContactKeyServiceImpl(keyService,
                publicExternalKeyService,
                getService(ContactService.class),
                getService(GuardRatifierService.class)));

        //Service for importing client created, custom keys
        registerService(KeyImportService.class, new KeyImportServiceImpl(keyRecoveryService,keyService,
                                                                         getService(KeyTableStorage.class),
                                                                         getService(PGPKeysStorage.class)));
        //Service for importing keys from command line
        registerService(ImportSystemPublicKey.class, new ImportSystemPublicKeyImpl());

        //Account creation Service
        registerService(AccountCreationService.class, new AccountCreationServiceImpl());
        openTrackers();
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#stop(org.osgi.framework.BundleContext)
     */
    @Override
    public void stop(BundleContext context) throws Exception {
        org.slf4j.LoggerFactory.getLogger(KeyManagementServicesImplActivator.class).info("Stopping bundle: {}", context.getBundle().getSymbolicName());
        unregisterService(KeyCreationService.class);
        Services.setServiceLookup(null);
    }
}
