package com.openexchange.guard.guestreset.internal;

import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.guestreset.GuestResetService;
import com.openexchange.guard.guestreset.exceptions.GuardGuestResetExceptionCodes;
import com.openexchange.guard.guestreset.osgi.Services;
import com.openexchange.guard.guestreset.storage.TemporaryTokenService;
import com.openexchange.guard.guestreset.storage.Token;
import com.openexchange.guard.guestreset.storage.impl.TemporaryTokenServiceImpl;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;

public class GuestResetServiceImpl implements GuestResetService {

    @Override
    public void reset(String email, String host, boolean force, String ip) throws OXException {

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        GuardNotificationService notifier = Services.getService(GuardNotificationService.class);
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);

        // Check for scanning for valid email accounts.
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        if (ip != null && antiabuse.isBad(ip)) {  // null ip from command line
            throw GuardAuthExceptionCodes.LOCKOUT.create();
        }

        GuardKeys keys = keyService.getKeys(email);
                // Check to make sure keys found, Guest account, and if recovery exists
        if (keys == null || keys.getContextid() > 0) {
            if (ip != null){
                antiabuse.addBad(ip);  // If account not found, add to lockout
            }
            throw GuardGuestResetExceptionCodes.UNKNOWN_GUEST_ACCOUNT.create(email);
        }
        if (keys.isRecoveryAvail() && !force) {
            throw GuardGuestResetExceptionCodes.RECOVERY_EXISTS.create(email);
        }

        int templId = guardConfigService.getIntProperty(GuardProperty.templateID, keys.getContextid() > 0 ? keys.getUserid() : -1, keys.getContextid());

        // Store resetId here
        TemporaryTokenService tokenService = new TemporaryTokenServiceImpl();
        Token token = new Token (email, keys.getUserid(), keys.getContextid());
        tokenService.storeToken(token);

        // Send email here
        JsonObject emailJson = mailCreatorService.getGuestResetEmail(email, token.getToken(), keys.getLanguage(), templId, host, keys.getUserid(), keys.getContextid());
        String url = "";
        url = guardConfigService.getProperty(GuardProperty.externalReaderURL, keys.getUserid(), keys.getContextid());
        if (url == null || url.isEmpty()) {
            url = "https://" + guardConfigService.getProperty(GuardProperty.externalReaderPath);
        }
        emailJson = mailCreatorService.addURL(emailJson, url + "?resetid=" + token.getToken() + "&user=" + email + "&lang=" + keys.getLanguage());
        notifier.send(emailJson, -1, keys.getContextid(), keys.getLanguage());

    }

    @Override
    public void doReset(JsonObject data, String ip) throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        TemporaryTokenService tokenService = new TemporaryTokenServiceImpl();
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        KeyCreationService keyCreationService = Services.getService(KeyCreationService.class);

        // Check for scanning for valid resetids.
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        if (ip != null && antiabuse.isBad(ip)) {  // null ip from command line
            throw GuardAuthExceptionCodes.LOCKOUT.create();
        }

        String tokenString = data.get("resetid").getAsString();
        if (tokenString == null) {
            throw GuardGuestResetExceptionCodes.BAD_TOKEN.create(tokenString);
        }


        Token token = tokenService.getToken(tokenString);
        // Check token found
        if (token == null) {
            antiabuse.addBad(ip);
            throw GuardGuestResetExceptionCodes.BAD_TOKEN.create(tokenString);
        }

        String email = data.get("email").getAsString();
        // Make sure email addresses agree
        if (!token.getEmail().toLowerCase().equals(email.toLowerCase())) {
            antiabuse.addBad(ip);
            throw GuardGuestResetExceptionCodes.BAD_EMAIL_ADDRESS.create(email);
        }

        String password = data.get("password").getAsString();
        if (password.length() < guardConfigService.getIntProperty(GuardProperty.minPasswordLength)) {
            throw GuardGuestResetExceptionCodes.MIN_PASSWORD_LENGTH.create(guardConfigService.getIntProperty(GuardProperty.minPasswordLength));
        }

        GuardKeys keys = keyService.getKeys(email);

        // Check to make sure keys found
        if (keys == null) {
            throw GuardGuestResetExceptionCodes.UNKNOWN_GUEST_ACCOUNT.create(email);
        }

        // Add new keys
        GuardKeys newkey = keyCreationService.create(keys.getUserid(), keys.getContextid(), keys.getEmail(), keys.getEmail(), password, keys.getLocale(), true, false);
        keyService.addNewKey(newkey);

        tokenService.deleteToken(token);  // Remove the token now


    }

}
