/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.guest.impl.converters;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Objects;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import com.openexchange.exception.OXException;
import com.openexchange.guard.guest.exceptions.GuardGuestExceptionCodes;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEMailMetadataService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEmailMetadata;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link MetaDataConverter}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class MetaDataConverter implements GuestItemConverter {

    private static final String                  DEFAULT_FOLDER        = "INBOX";
    private static final int                     DEFAULT_MESSAGE_FLAGS = 1 << 4; /** The recent flag. See {@link MailMessage} **/
    private final GuardGuestEMailMetadataService guardGuestEmailMetadataService;

    public MetaDataConverter(GuardGuestEMailMetadataService guardGuestEmailMetadataService) {
        this.guardGuestEmailMetadataService = Objects.requireNonNull(guardGuestEmailMetadataService, "GuardGuestEMailMetadataService service missing or null");
    }

    /**
     * Internal method to create a MimeMessage from the given stream
     *
     * @param inputStream The stream to create the MimeMessge from
     * @return The MimeMessage constructed from the given stream
     * @throws MessagingException
     */
    private MimeMessage createMimeMessage(InputStream inputStream) throws MessagingException {
        return new MimeMessage(Session.getDefaultInstance(new Properties()), inputStream);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.impl.converters.GuestItemConverter#convert(com.openexchange.guard.user.UserIdentity, java.io.InputStream, java.io.OutputStream, java.lang.String)
     */
    @Override
    public GuestItemConverterResult convert(UserIdentity guest, InputStream guardGuestData, OutputStream output, String objectId) throws OXException {
        // Check if meta data already exists
        if (guardGuestEmailMetadataService.get(guest.getOXGuardUser().getContextId(),
            guest.getOXGuardUser().getId(),
            objectId) != null) {
            return GuestItemConverterResult.NOT_CONVERTED_NOT_NECASSARY;
        }
        // Create
        MimeMessage msg;
        try {
            msg = createMimeMessage(guardGuestData);
            GuardGuestEmailMetadata metaData = guardGuestEmailMetadataService.createMetaDataFrom(objectId, DEFAULT_FOLDER, msg, DEFAULT_MESSAGE_FLAGS);
            guardGuestEmailMetadataService.insert(guest.getOXGuardUser().getContextId(), guest.getOXGuardUser().getId(), metaData);
        } catch (MessagingException e) {
            throw GuardGuestExceptionCodes.PROBLEM_WITH_CONVERSION.create(e);
        }
        return GuestItemConverterResult.NOT_CONVERTED_NOT_NECASSARY; // Nothing needed to be saved in output
    }

}
