/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.guest;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEmailMetadata;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link GuardGuestEmail} represents an Email object owned by a guest user
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class GuardGuestEmail implements Closeable {

    private final String            itemId;
    private MimeMessage             message;
    private UserIdentity            owner;
    private GuardGuestEmailMetadata metaData;
    private InputStream             messageStream;

    /**
     * Initializes a new {@link GuardGuestEmail}.
     *
     * @param owner The owner
     * @param itemId The ID of the item
     */
    public GuardGuestEmail(UserIdentity owner, String itemId) {
        this(owner, itemId, null);
    }

    /**
     * Initializes a new {@link GuardGuestEmail}.
     *
     * @param owner The owner
     * @param itemId The ID of the item
     * @param message The message
     */
    public GuardGuestEmail(UserIdentity owner, String itemId, InputStream messageStream) {
        this.owner = Objects.requireNonNull(owner, "owner must not be null");
        this.itemId = Objects.requireNonNull(itemId, "itemId must not be null");
        this.messageStream = messageStream;
    }

    /**
     * Gets the item's unique ID
     *
     * @return The unique ID of the item
     */
    public String getItemId() {
        return itemId;
    }

    /**
     * Gets the owner of the guest email
     *
     * @return The owner of the guest email.
     */
    public UserIdentity getOwner() {
        return owner;
    }

    /***
     * Gets the related message
     *
     * @return The related message
     */
    public MimeMessage getMessage() {
        return message;
    }

    /**
     * Gets the content of the related message, or null if there is no message
     *
     * @return The content of the message as byte array, or null of there is no message
     * @throws IOException
     * @throws MessagingException
     */
    public byte[] getContent() throws IOException, MessagingException {
        if (message != null) {
            try (ByteArrayOutputStream bytes = new ByteArrayOutputStream()) {
                message.writeTo(bytes);
                return bytes.toByteArray();
            }
        }
        return null;
    }


    /**
     * Gets the {@link InputStream} for the whole message content
     *
     * @return The {@link InputStream} for the message content
     */
    public InputStream getMessgeStream() {
        return this.messageStream;
    }

    /**
     * Sets the message
     *
     * @param message The message to set
     * @return this
     */
    public GuardGuestEmail setMessage(MimeMessage message) {
        this.message = message;
        return this;
    }

    public GuardGuestEmail setMessageStream(InputStream messageSteam) {
        this.messageStream = messageSteam;
        return this;
    }

    /**
     * Gets the {@link GuardGuestEmailMetadata}
     *
     * @return The meta data, or null if no meta data are present
     */
    public GuardGuestEmailMetadata getMetaData() {
        return metaData;
    }

    /**
     * Sets the {@link GuardGuestEmailMetadata}
     *
     * @param metaData The meta data to set
     * @return this
     */
    public GuardGuestEmail setMetaData(GuardGuestEmailMetadata metaData) {
        this.metaData = metaData;
        return this;
    }

    /*
     * (non-Javadoc)
     *
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() throws IOException {
        if (this.messageStream != null) {
            messageStream.close();
        }
    }
}
