/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database.update.shardUpdate;

import com.openexchange.exception.OXException;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.database.exception.GuardDatabaseExceptionCodes;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.server.ServiceLookup;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

/**
 * {@link ShardUpdaterServiceImpl}  Performs any updates required and registered on Guest shards
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.3
 */
public class ShardUpdaterServiceImpl implements ShardUpdaterService {

    ServiceLookup serviceLookup;
    GuardDatabaseService guardDatabaseService = null;
    ShardUpdateTask[] tasks;

    private static final String SELECT_TASK = "SELECT `success` FROM updates WHERE id = ?";
    private static final String UPDATE_TASK = "INSERT INTO updates (`id`, `success`) VALUES (?, ?) ON DUPLICATE KEY UPDATE `success` = ?";

    public ShardUpdaterServiceImpl (ServiceLookup serviceLookup, GuardDatabaseService guardDatabaseService, ShardUpdateTask...shardUpdateTasks) {
        this.serviceLookup = serviceLookup;
        this.guardDatabaseService = guardDatabaseService;
        this.tasks = shardUpdateTasks;
    }

    /**
     * Check updates table to see if task has been executed.
     *
     * @param shardId
     * @param name
     * @return
     * @throws OXException
     */
    private boolean checkDone (int shardId, String name) throws OXException {
        Connection connection = guardDatabaseService.getReadOnlyForShard(shardId);

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(SELECT_TASK);
            stmt.setString(1, name);
            resultSet = stmt.executeQuery();
            if (resultSet.next()) {
                return resultSet.getBoolean("success");
            }
            return false;
        } catch (SQLException e) {
            throw GuardDatabaseExceptionCodes.DB_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForShard(shardId, connection);;
        }
    }

    /**
     * Mark the task as success or failure
     *
     * @param shardId
     * @param name
     * @param success
     * @throws OXException
     */
    private void markDone (int shardId, String name, boolean success) throws OXException {
        Connection connection = guardDatabaseService.getWritableForShard(shardId);

        PreparedStatement stmt = null;
        try {
            stmt = connection.prepareStatement(UPDATE_TASK);
            stmt.setString(1, name);
            stmt.setBoolean(2, success);
            stmt.setBoolean(3, success);
            stmt.execute();
        } catch (SQLException e) {
            throw GuardDatabaseExceptionCodes.DB_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForShard(shardId, connection);
        }
    }

    @Override
    public void doUpdates() throws OXException {
        GuardShardingService shardingService = serviceLookup.getServiceSafe(GuardShardingService.class);

        List<Integer> shards = shardingService.getAllShards();
        for (Integer shard : shards) {
            doUpdates(shard);
        }
    }

    @Override
    public void doUpdates(int shard) throws OXException {
        for (ShardUpdateTask task: tasks) {
            if (!checkDone(shard, task.getName())) {
                markDone(shard, task.getName(), task.update(shard));
            }
        }

    }

}
