/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.cipher;

import java.security.Key;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardProperty;

/**
 * {@link GuardCipherService} to encrypt/decrypt data and generate random passwords and keys
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface GuardCipherService {

    /**
     * Encrypts the specified data with the specified {@link Key}
     * 
     * @param data The data to encrypt
     * @param secretKey The secret {@link Key} to use for the encryption
     * @return The encrypted data
     */
    String encrypt(String data, Key secretKey);

    /**
     * Encrypts the specified data with the specified base64 encoded key
     * 
     * @param data The data to encrypt
     * @param base64Key The base 64 key
     * @return The encrypted data
     */
    String encrypt(String data, String base64Key);

    /**
     * Encrypts the specified data with the specified password and the specified salt
     * 
     * @param data The data to encrypt
     * @param password The password to encrypt with
     * @param salt The salt to use
     * @return The encrypted data
     */
    String encrypt(String data, String password, String salt) throws OXException;

    /**
     * Decrypts the specified data with the specified password and the specified salt
     * 
     * @param data The data to decrypt
     * @param password The password to encrypt with
     * @param salt The salt to use
     * @return The decrypted data
     */
    String decrypt(String data, String password, String salt);

    /**
     * Decrypts the specified data with the specified password, the specified salt and the specified key length
     * 
     * @param data The data to decrypt
     * @param password The password to encrypt with
     * @param salt The salt to use
     * @param keyLength The key length
     * @return The decrypted data
     */
    String decrypt(String data, String password, String salt, int keyLength);

    /**
     * Decrypts the specified data with the specified base64 encoded key
     * 
     * @param data The data to decrypt
     * @param base64Key The base 64 encoded key
     * @return The decrypted data
     */
    String decrypt(String data, String base64Key);

    /**
     * Decrypts the specified data with the specified {@link Key}
     * 
     * @param data The data to decrypt
     * @param secretKey The {@link Key} to decrypt the data with
     * @return The decrypted data
     */
    String decrypt(String data, Key secretKey);

    /**
     * Generates, encodes in to base 64 and returns a random key
     * 
     * @return the generated random key in base 64 encoding
     */
    String getRandomKey();

    /**
     * Generates and returns a random password. The length of the password is defines in the {@link GuardProperty}
     * 
     * @param userId  Id of the user
     * @param cid  The context of the user
     * @return The random password
     * @throws OXException if the password cannot be generated
     */
    String generateRandomPassword(int userId, int cid) throws OXException;

}
