/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.autocrypt.impl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.autocrypt.database.AutoCryptKey;
import com.openexchange.guard.autocrypt.exceptions.AutocryptExceptionCodes;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.commons.trust.GuardKeySources;
import com.openexchange.pgp.keys.common.ModifyingPGPPublicKeyRing;

/**
 * {@link AutoCryptUtils}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.2
 */
public class AutoCryptUtils {

    private static Logger LOG = LoggerFactory.getLogger(AutoCryptUtils.class);

    /**
     * Wrap header to keep within 80 character limit for compliance
     * @param keydata
     * @return Wrapped data
     */
    public static String formatHeader(String keydata) {
        int length = keydata.length();
        int index = 0;
        StringBuilder sb = new StringBuilder();
        while (index < length) {
            sb.append("\r\n ");
            sb.append(keydata.substring(index, Math.min(index + 79, keydata.length())));
            index += 79;
        }
        return sb.toString();
    }

    /**
     * Strip the userIds out of a keyring with the exception of the one containing the specified email address
     * @param ring
     * @param email  Email address of the userId to maintain
     * @return
     */
    private static PGPPublicKeyRing removeOtherUserIDs (PGPPublicKeyRing ring, String email) {
        ModifyingPGPPublicKeyRing newRing = new ModifyingPGPPublicKeyRing();
        Iterator<PGPPublicKey> keys = ring.getPublicKeys();
        boolean found = false;
        while (keys.hasNext()) {
            PGPPublicKey key = keys.next();
            Iterator<byte[]> ids = key.getRawUserIDs();
            while (ids.hasNext()) {
                byte[] id = ids.next();
                String strId = new String(id, StandardCharsets.UTF_8);
                if (strId != null && !strId.contains(email)) {
                    key = PGPPublicKey.removeCertification(key, id);
                }
                if (strId.contains(email)) {
                    found = true;
                }
            }
            if (key != null) {
                newRing.addPublicKey(key);
            }
        }
        if (!found) {
            return null;  // No email matches, all IDs were stripped
        }
        return newRing.getRing();
    }


    /**
     * Base64 encode a pgp public key ring
     * @param ring
     * @return
     * @throws OXException
     */
    public static String createEncodedKeyString (PGPPublicKeyRing ring, String email) throws OXException {
        if (ring == null) {
            return "";
        }
        if (email != null) {
            ring = removeOtherUserIDs(ring, email);
        }
        if (ring == null) {
            return null;
        }
        try {
            return Base64.encodeBase64String(ring.getEncoded(true));
        } catch (IOException e) {
            throw OXException.general("Error encoding pgpKeyRing ", e);
        }
    }

    public static AutoCryptKey convertHeader (int userId, int cid, String header, long date) throws OXException {
        if (header == null) {
            return null;
        }
        String[] sections = header.split(";");
        String addr = "";
        String keydata = "";
        AutoCryptKey.Preference pref = AutoCryptKey.Preference.none;

        for (String section : sections) {
            String[] parts = section.split("=");
            if (parts.length >= 2) {
                switch (parts[0].trim()) {
                    case "addr":
                        addr = parts[1];
                        break;
                    case "prefer-encrypt":
                        if (parts[1].equals("mutual")) {
                            pref = AutoCryptKey.Preference.mutual;
                        }
                        break;
                    case "keydata":
                        keydata = parts[1];
                        break;
                }
            }
        }
        if (keydata.isEmpty() || addr.isEmpty()) {
            throw AutocryptExceptionCodes.MISSING_KEY_DATA.create();
        }
        byte[] keyByteData = Base64.decodeBase64(keydata);
        PGPPublicKeyRing keyRing;
        try {
            keyRing = new PGPPublicKeyRing(keyByteData, new BcKeyFingerprintCalculator());
        } catch (IOException e) {
            return null;
        }

        return new AutoCryptKey(userId, cid, addr, keyRing, date, pref);


    }

    /**
     * Check an autocrypt key to make sure the email address and key ids match the fromAddress
     * @param fromAddress
     * @param key
     * @return
     */
    public static boolean confirmEmail (AutoCryptKey key) {
        Iterator<PGPPublicKey> pubKeys = key.getPGPPublicKeyRing().getPublicKeys();
        while (pubKeys.hasNext()) {
            PGPPublicKey pubKey = pubKeys.next();
            Iterator<String> ids = pubKey.getUserIDs();
            while (ids.hasNext()) {
                String id = ids.next();
                if (id.toLowerCase().contains(key.getEmail())) {
                    return true;
                }
            }
        }
        return false;
    }

    public static String getHash (AutoCryptKey key) {
        try {
            return Hex.encodeHexString(MessageDigest.getInstance("SHA-256").digest(key.getPGPPublicKeyRing().getEncoded()));
        } catch (NoSuchAlgorithmException | IOException e) {
            LOG.error("Error creating hash for autoCrypt key", e);;
        }
        return null;
    }

    public static RecipKey convertToRecipKey (AutoCryptKey key) {
        RecipKey recipKey = new RecipKey(key.getVerified() ?
                GuardKeySources.AUTOCRYPT_KEY_USER_VERIFIED :
                GuardKeySources.AUTOCRYPT_KEY
            );
        recipKey.setEmail(key.getEmail());
        recipKey.setPgp(true);
        recipKey.setPGPPublicKeyRing(key.getPGPPublicKeyRing());
        recipKey.setUserid(key.getUserId());
        recipKey.setCid(key.getCid());
        return recipKey;
    }


}
