/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2017 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('Settings - Key Management');

Before(async function (users) {
    await users.create();
    await users.create();  // Second user for revoke test
});

After(async function (users) {
    await users.removeAll();
});

const assert = require('chai').assert;

Scenario('View details of your keys', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings 

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    // Check yourkeys list has at least two keys
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('#userkeytable');
    I.waitForVisible('.userpgpdetail');
    I.seeNumberOfVisibleElements('tr', 2);

    // Check key details
    I.click('.userpgpdetail');
    I.waitForVisible('.keystart');
    let keyDetail = await I.grabTextFrom('#privKeyDetail .modal-body');
    assert.include(keyDetail, 'Fingerprint', 'Key Detail includes fingerprint');

    // Check able to check signatures
    I.click('.btn[data-action="signatures"]');
    I.wait(1);
    I.see('Public Keys Detail', '#pkeyDetail .modal-title');
    I.seeNumberOfVisibleElements('tr', 2);
    I.see('Positive', '[title="Positive certification of a User ID"]');

    // Close all
    I.click('#pkeyDetail [data-action="done"]');
    I.click('#privKeyDetail [data-action="done"]');
    I.click('.btn[data-action="done"]');
    I.logout();

});

Scenario('Upload public keys and verify available', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;
    await I.setupUser(o, true);  // need advanced settings 

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    I.waitForVisible('.btn[name="recipients"]');
    I.click('.btn[name="recipients"]');

    I.waitForElement('.publicKeyDiv [name="publicKey"]');

    I.executeAsyncScript(function (done) {
        $('[name="publicKey"]').on('change', function () {
                var files = this.files;
                if (files.length > 0) {
                    require(['oxguard/pgp/uploadkeys'], function (uploader) {
                        uploader.uploadExternalKey(files)
                        .always(function () {
                            
                        });
                    });
                }
        });
        done();
    });
    I.attachFile('.publicKeyDiv [name="publicKey"]', 'e2e/testFiles/email-example-com.asc');
    I.wait(1);

    // Confirm uploaded, and check key details
    I.click('.refreshkeys');
    I.wait(1);
    I.seeNumberOfVisibleElements('tr', 1);
    I.see('Test User ', 'td');
    I.click('.oxguard_pubkeylist a[ids="6be6dc8 b2f16762"]');
    I.waitForVisible('#pkeyDetail .modal-title');
    I.see('Public Keys Detail', '.modal-title');
    I.seeNumberOfVisibleElements('#keyDetail', 2);

    I.click('Signatures');
    I.waitForVisible('.signatureTable');
    I.seeNumberOfVisibleElements('tr', 2);
    I.see('Positive', '.signatureTable');
    I.click('.modal[data-point="oxguard/pgp/signatureview"] [data-action="done"]');
    I.wait(1);

    I.click('#pkeyDetail [data-action="done"]');
    I.click('#pkeyList [data-action="done"]');

    // Start a mail to see if key properly loaded
    I.openApp('Mail');
    I.waitForVisible('.io-ox-mail-window .window-body .classic-toolbar');
    I.clickToolbar('Compose');
    I.waitForVisible('.window-blocker.io-ox-busy');
    I.waitForInvisible('.window-blocker.io-ox-busy');
    I.waitForVisible('.io-ox-mail-compose textarea.plain-text,.io-ox-mail-compose .contenteditable-editor');

    // Mark as secure
    I.click('Security');
    I.waitForVisible('.dropdown.open a[data-name="encrypt"]');
    I.click('Encrypt');
    I.waitForVisible('.fa.fa-2x.encrypted');

    // Add uploaded key email address and confirm key found and properly labeled
    I.fillField('.io-ox-mail-compose div[data-extension-id="to"] input.tt-input', 'email@example.com');
    I.fillField('[name="subject"]', 'test');
    I.waitForVisible('.fa-key');
    I.moveCursorTo('i.fa-key', 5, 5);
    I.see('Uploaded', '.tooltip');

    // Cleanup
    I.click('.btn[data-action="discard"]');
    I.waitForElement('.btn[data-action="delete"]');
    I.click('.btn[data-action="delete"]');
    I.waitForInvisible('.modal-backdrop');
    I.logout();
});

Scenario('Private Key Download and Upload', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings 

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    // First, download the current private key
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.delPriv');
    // Get the keyId from table and download
    let keyId = await I.grabAttributeFrom('.delPriv', 'value');
    let downloadedKey = await I.downloadPrivateKey(keyId, userdata);

    // OK, delete the key
    I.click('.delPriv');
    I.waitForVisible('.modal[data-point="oxguard/settings/deletePrivate"]');
    I.click('.btn[data-action="delete"]');
    I.waitForVisible('#deletepass');
    I.insertCryptPassword('#deletepass', userdata);
    I.click('.btn[data-action="delete"]');

    // Upload dialog should appear now
    I.waitForVisible('.modal[data-point="oxguard/settings/uploadPrivate"]');

    // We have to manualy perform the upload here
    await I.executeAsyncScript(function(downloadedKey, userdata, done) {
        var formData = new FormData();
        formData.append('key', downloadedKey);
        formData.append('keyPassword', userdata.password);
        formData.append('newPassword', userdata.password);
        var url = ox.apiRoot + '/oxguard/keys?action=upload&respondWithJSON=true&session=' + ox.session;
        $.ajax({
            url: url,
            type: 'POST',
            data: formData,
            processData: false, // tell jQuery not to process the data
            contentType: false, // tell jQuery not to set contentType
            success: function (data) {
                done(data);
            },
            fail: function(e) {
                done(e);
            }
        });
    }, downloadedKey, userdata);

    // Cancel dialog since uploaded already
    I.click('.btn[data-action="cancel"]');

    // Confirm key is now there and same
    I.click('#refreshuserkeys');
    I.wait(1);
    I.waitForVisible('.delPriv');
    let checkKeyId = await I.grabAttributeFrom('.delPriv', 'value');
    assert.equal(checkKeyId[0], keyId[0], 'Key Ids should match');

    // Cleanup
    I.click('.btn[data-action="done"]');
    I.logout();

});

Scenario('Revoke Private Key', async function (I, users) {

    const o = {
        user: users[1]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings 

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    // Revoke the key
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.delPriv');
    I.click('.delPriv');
    I.waitForVisible('.modal[data-point="oxguard/settings/deletePrivate"]');
    I.click('.btn[data-action="revoke"]');
    I.waitForVisible('#revokepass');
    I.insertCryptPassword('#revokepass', userdata);
    I.selectOption('#settings-revokereason', 'KEY_SUPERSEDED');
    I.click('.btn[data-action="revoke"]');

    // Confirm drawn as revoked
    I.wait(1);
    let property = await I.grabCssPropertyFrom('tr', 'color');
    assert.isTrue((property[0].indexOf('rgb(255') > -1) || (property[0].indexOf('rgba(255') > -1), 'Row color should now be red');

    // Check detail
    I.click('.userpgpdetail');
    I.waitForVisible('#privKeyDetail .modal-title');
    let keyDetail = await I.grabTextFrom('#privKeyDetail .modal-body');
    assert.include(keyDetail, 'Revoked', 'Key Detail shows revoked');

    // Cleanup
    I.click('#privKeyDetail [data-action="done"]');
    I.click('.btn[data-action="done"]');
    I.logout();

});
