/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 * Author Greg Hill <greg.hill@open-xchange.com>
 */
define('pgp_local/keyDetails', [
    'gettext!oxguard',
    'settings!oxguard'
], function (gt, settings) {
    'use strict';

    var keyring = window.keyRing;

    function getLocalKeysTable() {
        // Switch on storage type
        return getLocalStoreKeysTable();
    }

    function getLocalStoreKeysTable() {
        keyring = window.keyRing;
        var keys = keyring ? keyring.privateKeys.keys : [];
        if (keys.length === 0) {
            if (settings.get('localKeys')) {
                settings.clear('localKeys');
                settings.save();  // Make sure settings is cleared
            }
            return $('#localuserkeytable').html('<table id="userkeytable"><tr><td>' + gt('No Keys Found') + '</td></tr></table>');
        }
        if (!settings.get('localKeys')) settings.set('localKeys', true).save();
        var newtable = $('<table id="localuserkeytable">');
        var headers = $('<th>' + gt('Key ID') + '</th><th>' + gt('Details') + '</th><th>' + gt('Delete') + '</th><th>' + gt('Upload Public Key') + '</th>');
        newtable.append(headers);

        for (var grp = 0; grp < keys.length; grp++) {
            var key = keys[grp];
            key.getKeyIds();
            var dontUse = (new Date(key.getExpirationTime()).getTime() < new Date().getTime()) || key.revocationSignature !== null;
            newtable.append(tableRow(formatKey(key.primaryKey, true, dontUse)));
            var subKeys = key.getSubkeyPackets();
            for (var sub = 0; sub < subKeys.length; sub++) {
                var subkey = subKeys[sub];
                newtable.append(tableRow(formatKey(subkey, false, dontUse)));
            }
        }
        return newtable;
    }

    function formatKey(key, master, dontUse, userIds) {
        var data = {
            expired: dontUse,
            revoked: key.revocationSignature !== undefined && key.revocationSignature !== null,
            masterKey: master,
            Key: key.fingerprint === null ? ' ' : key.fingerprint.substring(key.fingerprint.length - 8),
            fingerPrint: key.fingerprint,
            creationTime: key.created ? key.created : '',
            usedForEncryption: key.algorithm.indexOf('encrypt') > 0,
            usedForSigning: key.algorithm.indexOf('sign') > 0,
            ids: userIds ? userIds : [],
            privateKey: key.encrypted || key.isDecrypted,
            localKey: true
        };
        return data;
    }

    function tableRow(data) {
        var tr = (data.expired) ? $('<tr style="color:red;">') : $('<tr>');
        var td1 = $('<td style="text-align:left;">');
        if (data.masterKey === false) {
            td1.append('<i class="fa fa-level-up fa-rotate-90" style="padding-right:7px; vertical-align:bottom;"></i>');
        }
        td1.append($('<span style="padding-left:5px;">' + data.Key + '</span>'));
        tr.append(td1);
        var td4 = $('<td>');
        var td5 = $('<td>');
        if (data.masterKey) {
            var search = $('<a class="localpgpdetail" ids="' + data.fingerPrint + '" href="#"><i class="fa fa-search"/></a>');
            search.click(function (e) {
                prepareKeys(data.fingerPrint);
                e.preventDefault();
            });
            td4.append(search);
            var del = $('<a href="#" value="' + data.fingerPrint + '" class="delLocalPriv"><i class="fa fa-trash"/></a>');
            del.click(function (e) {
                deleteKey(data.fingerPrint);
                e.preventDefault();
            });
            td5.append(del);
        }
        var td7 = $('<td>');
        var upload = $('<a class="keyupload" href="#" value="' + data.fingerPrint + '"><i class="fa fa-upload"/></a>');
        upload.click(function (e) {
            uploadPublic(data.fingerPrint);
            e.preventDefault();
        });
        td7.append(data.usedForEncryption ? upload : '');  // Only keys with Encryption flag should be uploaded to Guard
        tr.append(td4).append(td5).append(td7);
        return (tr);
    }

    function deleteKey(fingerprint) {
        require(['oxguard/core/confirmView', 'pgp_local/register'], function (confirm, main) {
            confirm.open(gt('Please confirm you want to delete this key.'))
            .done(function () {
                var newPrivKeys = [];
                for (var i = 0; i < keyring.privateKeys.keys.length; i++) {
                    var key = keyring.privateKeys.keys[i];
                    if (key.primaryKey && key.primaryKey.fingerprint === fingerprint) {
                        // delete
                    } else {
                        newPrivKeys.push(key);
                    }
                }
                keyring.privateKeys.keys = newPrivKeys;
                main.storeKeys();
                $('#refreshlocalkeys').click();
            });
        });

    }

    function uploadPublic(fingerprint) {
        for (var i = 0; i < keyring.privateKeys.keys.length; i++) {
            var key = keyring.privateKeys.keys[i];
            if (key.primaryKey && key.primaryKey.fingerprint === fingerprint) {
                upload(keyring.privateKeys.keys[i].toPublic().armor());
            }
        }
    }

    function upload(key) {
        require(['oxguard/pgp/uploadkeys'], function (uploader) {
            uploader.uploadPublicKey(key);
        });
    }

    function prepareKeys(fingerprint) {
        var keyToShow;
        for (var i = 0; i < keyring.privateKeys.keys.length; i++) {
            var key = keyring.privateKeys.keys[i];
            if (key.primaryKey && key.primaryKey.fingerprint === fingerprint) {
                keyToShow = key;
            }
        }
        if (!keyToShow) return;
        var keys = [];
        keyToShow.getKeyIds();
        var dontUse = (new Date(keyToShow.getExpirationTime()).getTime() < new Date().getTime()) || keyToShow.revocationSignature !== null;
        var primary = formatKey(keyToShow.primaryKey, true, dontUse, keyToShow.getUserIds());
        keys.push(primary);
        primary.Expiration = (new Date(keyToShow.getExpirationTime()).toDateString());
        var subKeys = keyToShow.getSubkeyPackets();
        for (var sub = 0; sub < subKeys.length; sub++) {
            var subkey = subKeys[sub];
            keys.push(formatKey(subkey, false, dontUse));
        }
        showDetail(keys);
    }

    function showDetail(keys) {
        require(['oxguard/pgp/detailView', 'oxguard/pgp/keyDetails'], function (view, publickeys) {
            var keydetail = publickeys.keyDetail(keys);
            view.open(keydetail, { localKeys: true }, true);
        });
    }

    return {
        getLocalKeysTable: getLocalKeysTable
    };

});

