/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/settings/views/resetPasswordView', [
    'io.ox/backbone/views',
    'io.ox/core/extensions',
    'io.ox/backbone/mini-views',
    'io.ox/backbone/views/modal',
    'io.ox/core/settings/util',
    'io.ox/core/yell',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'gettext!oxguard'
], function (views, ext, mini, ModalView, util, yell, og_http, core, gt) {

    'use strict';

    var POINT = 'oxguard/settings/reset',
        INDEX = 0;

    function open() {
        return openModalDialog();
    }

    function openModalDialog() {

        return new ModalView({
            async: true,
            focus: 'input[name="active"]',
            point: POINT,
            title: gt('Reset Password'),
            width: 640
        })
        .inject({
            doReset: function () {
                return doReset();
            }
        })
        .build(function () {

        })
        .addCancelButton()
        .addButton({ label: gt('Reset'), action: 'reset' })
        .on('reset', function () {
            this.doReset().done(this.close).fail(this.idle).fail(yell);
        })
        .open();
    }

    ext.point(POINT).extend(
        //
        // Password Boxes
        //
        {
            index: INDEX += 100,
            id: 'switch',
            render: function () {

                this.$body.append(
                    $('<p>').append(gt('This will reset your %s password, and send you a new password to the configured email address.', window.oxguarddata.productName))
                );
            }
        }
    );

    function doReset() {
        var def = $.Deferred();
        var performReset = function (primaryEmail) {
            var waitdiv = $('<div class="og_wait" id="keygen"></div');
            waitdiv.append('<i class="fa-key icon-key fa fa-spin icon-spin"/><span>' + gt('Generating key, Please wait') + '</span>');
            $('.window-container-center').append(waitdiv);
            waitdiv.show();
            var param = '&userid=' + ox.user_id + '&cid=' + ox.context_id +
            '&default=' + encodeURIComponent(primaryEmail) + '&lang=' + ox.language;
            core.metrics('settings', 'reset-password');
            og_http.get(ox.apiRoot + '/oxguard/login?action=reset', param)
            .done(function (data) {
                $('.og_wait').remove();
                core.removeAuth();
                require(['io.ox/core/notifications'], function (notify) {
                    if (data.indexOf('primary') >= 0) {
                        notify.yell('success', gt('A new password has been sent to your email address.'));
                        window.oxguarddata.passcode = 'Password Needed';
                        def.resolve();
                        return;
                    }
                    if (data.indexOf('ok') >= 0) {
                        notify.yell('success', gt('A new password has been sent to your secondary email address.'));
                        window.oxguarddata.passcode = 'Password Needed';
                        def.resolve();
                        return;
                    }
                    if (data.trim() === 'NoRecovery') {
                        notify.yell('error', gt('No password recovery is available.'));
                        def.reject();
                        return;
                    }
                    if (data.indexOf('FailNotify') >= 0) {
                        notify.yell('error', gt('Password reset, but unable to send to your email address.'));
                        def.reject();
                        return;
                    }
                    if (data.indexOf('NoSecondary') >= 0) {
                        notify.yell('error', gt('Unable to find email address to send password reset.'));
                        def.reject();
                        return;
                    }
                    notify.yell('error', gt('Unable to reset your password'));
                    def.reject();
                });
            })
            .fail(function () {
                $('.og_wait').remove();
                require(['io.ox/core/notifications'], function (notify) {
                    notify.yell('error', gt('Unable to reset your password'));
                });
                def.reject();
            });
        };

        require(['io.ox/core/api/account'], function (api) {
            api.getPrimaryAddress().done(function (data) { // Get primary address as backup
                var primaryEmail = data[1];
                performReset(primaryEmail);
            })
            .fail(function () {
                performReset(ox.user);  // Possible guest, no primary email
            });
        });
        return def;

    }

    return {
        open: open
    };

});
