/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/mail/options_new', [
    'oxguard/mail/oxguard_mail_compose_core',
    'oxguard/mail/util',
    'oxguard/util',
    'settings!oxguard',
    'gettext!oxguard',
    'less!oxguard/mail/style'
], function (core, util, coreUtil, settings, gt) {
    'use strict';

    var lockclass = 'icon-lock icon-2x fa fa-lock fa-2x oxguard_lock';

    function createlock(baton) {
        baton.view.listenTo(baton.view.securityModel, 'change:encrypt', function () {
            setLock(baton);
        });
        setLock(baton);
    }

    function setLock(baton) {
        if (baton.view.securityModel.get('encrypt')) {
            core.lock(baton);
        } else {
            core.unlock(baton);
        }
    }

    function warnHTML(baton) {
        var format = baton.view.securityModel.get('PGPFormat');
        var encrypt = baton.view.securityModel.get('encrypt');
        if (format !== 'inline' || !encrypt) return;  // Don't display warning if not Inline or not encrypted
        if (baton.config.get('editorMode') === 'text') return;
        require(['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true });
            dialog.header($('<h4>').text(gt('HTML Formatted Email')));
            var body = $('<p>');
            body.append(gt('HTML formatted emails may cause issues with PGP inline.  Click OK to change to Plain Text formatted Emails.'));
            dialog.getBody().append(body);
            dialog
            .addPrimaryButton('ok', gt('OK'), 'okx')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('ok', function () {
                baton.config.set('editorMode', 'text');
                baton.config.on('change:editorMode', function () {
                    if ((baton.model.get('editorMode') === 'html') && (baton.view.securityModel.get('PGPFormat') === 'inline')) {
                        var dialog2 = new dialogs.CreateDialog({ width: 450, height: 300, center: true });
                        dialog2.header($('<h4>').text(gt('HTML Formatted Email')));
                        var warn = $('<p>').append(gt('HTML format is not recommended when using PGP Inline'));
                        dialog2.getBody().append(warn);
                        dialog2
                        .addPrimaryButton('ok', gt('OK'), 'okx')
                        .show();
                    }
                });
            })
            .show();
        });
    }

    function getType(current) {
        if (current === null || current === undefined) {  // if no decoded data, then return default
            if (settings.get('defaultInline')) {
                return ('inline');
            } return ('mime');
        }
        if (current.pgpFormat === 'pgpmime') return ('mime');
        return ('inline');
    }

    function createOptions(baton, count) {
        if (window.oxguarddata.settings.oxguard === undefined) {  // If we don't yet have response from the Guard backend, check every second then handle
            if (count === undefined) count = 0;
            if (count > 10) return;  // limit our loop to 10 seconds
            window.setTimeout(function () {
                count++;
                createOptions(baton, count);
            }, 1000);
        }

        if (window.oxguarddata.action === 'sendfile') { // If it is going to be sending already encrypted file, then don't add encryption options
            window.oxguarddata.action = null;
            return;
        }
        if (window.oxguarddata.sentfolder === undefined) { // Get sent folder data for sending
            require(['settings!io.ox/mail']).done(function (settings) {
                window.oxguarddata.sentfolder = settings.get('folder/sent');
                window.oxguarddata.draftfolder = settings.get('folder/drafts');
            });
        }

        var securityModel = baton.view.securityModel;

        // If this is a secure reply, then disable all of the options to make it plain text
        if ((window.oxguarddata.action === 'secureReply') || (window.oxguarddata.action === 'fileAttach') || (util.hasEncryptedAttachment(baton) > 0)) {
            if (window.oxguarddata.replyInline === true) {
                securityModel.setInline();
                window.oxguarddata.replyInline = null;
            }
            if (securityModel.get('PGPFormat') === undefined) { // If not previously defined
                securityModel.set('PGPFormat', getType(window.oxguarddata.current));
            }
            securityModel.set('encrypt', true);
            if (window.oxguarddata.settings.oxguard === false || window.oxguarddata.settings.secureReply === true) {
                baton.view.$el.find('#statuslock')
                    .attr('class', lockclass)
                    .click(false)
                    .attr('title', gt('Reply must be encrypted'));
                baton.view.$el.find('[data-name="encrypt"]').click(false);
            }
            window.oxguarddata.action = null;
        }

        var headers = baton.model.get('headers');
        if (headers && (headers['X-OxGuard-Sign'] === 'true')) {
            securityModel.set('PGPSignature', true);
        }

        createlock(baton);

        if (baton.config.get('newmail') === true && coreUtil.hasGuardMailCapability()) {  // New email doesn't have recips yet.  Check to see if there are defaults
            // If default is to use guard, go ahead and lock
            if (settings.get('defaultEncrypted')) {
                securityModel.set('encrypt', true);
                core.lock(baton);
            } else securityModel.set('encrypt', false);
            if (settings.get('defaultInline')) {
                if (settings.get('defaultEncrypted')) {
                    securityModel.setInline();
                    warnHTML(baton);
                } else {
                    securityModel.setInline();
                }
            } else securityModel.setMime();
            baton.config.unset('newmail');

            if (settings.get('defaultSign')) {
                securityModel.set('PGPSignature', true);
            }
        }

        securityModel.on('change:PGPFormat change:encrypt', function () {
            warnHTML(baton);
        });

        securityModel.on('change:PGPFormat', function () {
            securityModel.set('encrypt', true);
        });

        securityModel.on('change:PubKey', function () {
            if (securityModel.get('PubKey')) {
                attachPubKey(baton);
            }
        });

    }
    // Attach the users public key to the email
    function attachPubKey(baton) {
        var attachments = baton.model.get('attachments');
        var securityModel = baton.view.securityModel;
        for (var i = 0; i < attachments.models.length; i++) {
            if (attachments.models[i].get('guard') === 'pubkey') {
                securityModel.set('PubKey', false);
                return;
            }
        }
        require(['oxguard/mail/pubkey', 'io.ox/core/notifications', 'io.ox/mail/compose/api'], function (pk, notify, composeApi) {
            pk.getPubKeyAttachment()
            .done(function (key) {
                try {
                    var file = new Blob([key.content], { type: 'application/pgp-keys' });
                    file.name = key.filename;
                    composeApi.space.attachments.add(baton.model.get('id'), { file: file })
                    .then(function (data) {
                        baton.model.attachFiles(data);
                    });
                } catch (e) {
                    console.error(e);
                    notify.yell('error', gt('Unable to attach your public key'));
                }

            })
            .fail(function () {
                notify.yell('error', gt('Unable to attach your public key'));
            })
            .always(function () {
                securityModel.set('PubKey', false);
            });
        });
    }

    return {
        createOptions: createOptions,
        warnHTML: warnHTML
    };
});
