/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/files/register', [
    'io.ox/core/extensions',
    'io.ox/core/capabilities',
    'oxguard/oxguard_core',
    'io.ox/core/folder/api',
    'oxguard/files/util',
    'io.ox/core/dropzone',
    'oxguard/auth',
    'gettext!oxguard',
    'io.ox/backbone/views/actions/util'
], function (ext, capabilities, oxguard, folderAPI, util, dropzone, auth_core, gt, actionsUtil) {

    'use strict';

    // For mobile, add a decrypt button
    ext.point('io.ox/files/details').extend({
        index: 500,
        id: 'downloadGuardsmall',
        draw: function (baton) {
            if (_.device('small')) {
                if (_.device('ios')) return;   // ios doesn't allow downloading.
                if (baton.data.ogfile === true) {
                    var download = $('<a href="#" class="btn btn-default">' + gt('Download decrypted') + '</a>');
                    download.click(function (ev) {
                        ev.preventDefault();
                        auth_core.authorize(baton).then(function (data) {
                            require(['oxguard/files/downloader'], function (downloader) {
                                metrics('guard', 'decrypt & download');
                                downloader.viewFile(baton, 'download', data);
                            });
                        });
                    });
                    $(this).append(download);
                }
            }
        }
    });

    ///////////  Main toolbar in drive

    //////   Links
    // Add link for add and encrypt local file
    new actionsUtil.Action('io.ox/files/toolbar/new', {
        index: 110,
        id: 'uploadEncrypt',
        title: gt('Add and encrypt local file'),
        ref: 'io.ox/files/actions/uploadEncrypt'
    });

    ext.point('io.ox/files/toolbar/links').extend({
        id: 'remencrypt',
        index: 550,
        title: gt('Remove encryption'),
        ref: 'oxguard/remencrypt',
        mobile: 'lo',
        section: 'guard'
    });

    ext.point('io.ox/files/toolbar/links').extend({
        id: 'encrypt',
        index: 550,
        title: gt('Encrypt'),
        ref: 'oxguard/encrypt',
        mobile: 'lo',
        section: 'guard'
    });

    ext.point('io.ox/files/toolbar/links').extend({
        id: 'sendcopy',
        index: 1200,
        prio: 'lo',
        mobile: 'lo',
        title: gt('Send by mail'),
        ref: 'oxguard/sendcopy',
        section: 'share'
    });

    // Mobile links
    ext.point('io.ox/files/links/inline').extend({
        id: 'encrypt',
        index: 1000,
        prio: 'hi',
        mobile: 'lo',
        title: gt('Encrypt'),
        ref: 'oxguard/encrypt'
    });

    ext.point('io.ox/files/links/inline').extend({
        id: 'remencrypt',
        index: 1000,
        prio: 'hi',
        mobile: 'lo',
        title: gt('Remove encryption'),
        ref: 'oxguard/remencrypt'
    });

    ext.point('io.ox/files/links/inline').extend({
        id: 'sendcopy',
        index: 1200,
        prio: 'hi',
        mobile: 'lo',
        title: gt('Send by mail'),
        ref: 'oxguard/sendcopy',
        section: 'share'
    });

    // Context menu

    ext.point('io.ox/files/listview/contextmenu').extend(
        {
            id: 'encrypt',
            index: 1450,
            ref: 'oxguard/encrypt',
            section: '25',
            title: gt('Encrypt')
        }
    );

    ext.point('io.ox/files/listview/contextmenu').extend(

        {
            id: 'decrypt',
            index: 1450,
            ref: 'oxguard/remencrypt',
            section: '25',
            title: gt('Remove encryption')
        }
    );

    // Change send copy to handle Guard emails
    ext.point('io.ox/files/actions/send').extend({
        id: 'og_stop_send',
        index: 1,
        matches: function (e) {
            if (e.proceed) return;
            if (e.collection.has('some', 'items') && !_.isEmpty(e.data)) {
                var toCheck = e.models === null ? e.model : e.models;
                if (util.hasEncrypted(toCheck)) {
                    e.stopPropagation();
                    return false;
                }
            }
        }
    });

    //////////////  Viewer
    /// Links

    // Extension to handle direct link files. Will not be used for opening list of files with selection
    ext.point('io.ox/core/viewer/main').extend({
        id: 'guardAuthCheck',
        index: 10,
        perform: function (baton) {
            var def = $.Deferred();
            if (baton.data.selection) {
                return def.resolve();  // Not handling here
            }
            var files = baton.data.files;
            if (!files || files.length === 0) {
                return def.resolve();
            }
            var file = files[0];
            if (isOGFile(file)) {
                auth_core.authorize(file, undefined, true)
                .then(function (auth) {
                    var file_options = { params: {
                        cryptoAction: 'Decrypt',
                        cryptoAuth: auth,
                        session: ox.session
                    } };
                    // fix for #58378
                    for (var i = 0; i < baton.data.fileList.length; i++) {
                        // fix for #58617
                        if (baton.data.fileList[i] instanceof Backbone.Model) {
                            baton.data.fileList[i].set('file_options', file_options);
                            baton.data.fileList[i].set('source', 'guardDrive');

                        } else {
                            baton.data.fileList[i].file_options = file_options;
                            baton.data.fileList[i].source = 'guardDrive';
                        }
                    }

                    def.resolve(baton);
                }, def.reject);
                return def;
            }
            return def.resolve();

        }
    });

    ext.point('io.ox/core/viewer/actions/toolbar/popoutstandalone').extend({
        id: 'popoutstandalone_guardcheck',
        index: 1,
        device: '!smartphone',
        matches: function (baton) {
            var model = baton.model;
            return model.get('group') !== 'localFile' && !baton.context.standalone && model.isEncrypted();
        },
        action: function (baton) {
            if (baton.model.isEncrypted()) {
                actionsUtil.invoke('oxguard/popoutstandalone', baton);
                baton.preventDefault();
            }
        }
    });

    ext.point('io.ox/core/viewer/toolbar/links/drive').extend({
        id: 'GuardViewerEncrypt',
        index: 200,
        title: gt('Encrypt'),
        ref: 'oxguard/encrypt',
        mobile: 'lo'
    });

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend({
        id: 'GuardViewerRemEncrypt',
        index: 200,
        title: gt('Remove encryption'),
        ref: 'oxguard/remencrypt',
        prio: 'lo',
        mobile: 'lo'
    });

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend({
        id: 'GuardViewerDownload',
        index: 201,
        title: gt('Download decrypted'),
        ref: 'oxguard/download',
        section: 'export',
        mobile: 'lo'
    });

    ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend({
        id: 'GuardViewerDownloadEncr',
        index: 202,
        title: gt('Download encrypted'),
        ref: 'oxguard/downloadEncrypted',
        section: 'export',
        mobile: 'lo'
    });

    // Versions

    ext.point('io.ox/files/versions/links/inline').extend({
        id: 'GuardView',
        index: 90,
        title: gt('View this version'),
        ref: 'oxguard/versionView'
    });

    ext.point('io.ox/files/versions/links/inline').extend({
        id: 'GuardDownload',
        index: 200,
        title: gt('Download decrypted'),
        ref: 'oxguard/download'
    });

    //// Actions

    // Upload new encrypted file

    function isTrash(baton) {
        var folderId, model;
        if (baton.app) {
            folderId = baton.app.folder.get();
        } else if (baton.folder_id !== undefined) {
            folderId = baton.folder_id;
        } else if (baton.data) {
            folderId = baton.data.folder_id;
        }
        model = folderAPI.pool.getModel(folderId);
        return model ? folderAPI.is('trash', model.toJSON()) : false;
    }

    // Upload new encrypted file
    new actionsUtil.Action('io.ox/files/actions/uploadEncrypt', {
        capabilities: 'guard-drive && !guest',
        matches: function (baton) {
            if (_(['14', '15']).contains(baton.folder_id)) return false;
            if (isTrash(baton)) return false;
            return true;
        },
        action: function (baton) {
            ensureSetup().then(function () {
                baton.file_options = { 'params': { 'cryptoAction': 'Encrypt' } };
                actionsUtil.invoke('io.ox/files/actions/upload', baton);
            });
        }
    });

    new actionsUtil.Action('oxguard/downloadEncrypted', {
        matches: function (baton) {
            if (baton.first()) {
                if (isOGFile(baton.first())) {
                    return (true);
                }
            }
            return false;
        },
        action: function (baton) {
            baton.array().map(function (file) {
                // For encrypted, wipe the cryptoAuth and cryptoAction
                if (file && file.params) {
                    file.params = {};
                }
            });
            actionsUtil.invoke('io.ox/files/actions/download', baton);

        }
    });

    // Disable publication of Guard files
    ext.point('io.ox/files/actions/getalink').extend({
        index: 10,
        id: 'guardCheck',
        matches: function (baton) {
            if (isOGFile(baton.first())) {
                baton.stopPropagation();
                return false;
            }
        }
    });

    ext.point('io.ox/files/actions/invite').replace('default', function (original) {
        return {
            action: function (baton) {
                if (isOGFile(baton.first())) {
                    require(['oxguard/auth'], function (auth_core) {
                        var options = {
                            optPrompt: gt('Please re-enter your %s password.', window.oxguarddata.productName),
                            minSingleUse: true
                        };
                        auth_core.authorize(baton, options)
                        .done(function () {
                            original.action(baton);
                        })
                        .fail(function (e) {
                            require(['io.ox/core/notifications'], function (notify) {
                                notify.yell('error', e);
                            });
                        });
                    });
                } else {
                    original.action(baton);
                }
            }
        };
    });

    // Replacement popout for Guard
    new actionsUtil.Action('oxguard/popoutstandalone', {
        collection: 'one',
        id: 'guardPopout',
        matches: function (baton) {
            var currentApp = ox.ui.App.getCurrentApp().getName();
            // detail is the target of popoutstandalone, no support for mail attachments
            return (currentApp !== 'io.ox/files/detail' && baton.model.isEncrypted());
        },
        action: function (e) {
            ox.launch('io.ox/files/detail/main').done(function () {
                var node = this.getWindowNode();
                this.setTitle(e.data.filename);
                var app = this;
                ox.load(['io.ox/mail/actions/viewer']).done(function (action) {
                    action({ files: [e.data], app: app, container: node, standalone: true });
                });
            });
        }
    });

    // View link within versions
    new actionsUtil.Action('oxguard/versionView', {
        matches: function (baton) {
            if (!baton.isViewer) { return false; }
            return (isOGFile(baton.first()));
        },
        action: function (baton) {
            viewFile(baton);
        }
    });

    new actionsUtil.Action('oxguard/sendcopy', {
        capabilities: 'guard-drive && webmail',
        collection: 'some',
        matches: function (baton) {
            //all files must be encrypted
            var toCheck = baton.models === null ? baton.model : baton.models;
            return util.hasEncrypted(toCheck) && util.isNotAttachment(toCheck);
        },
        action: function (baton) {
            window.oxguarddata.action = 'fileAttach';
            baton.proceed = true;
            actionsUtil.invoke('io.ox/files/actions/send', baton);
        }
    });

    function okToDecrypt(baton, data) {
        var virtual = _.contains(['14', '15'], data.id);
        return baton.collection.has('some', 'modify', 'items') && util.encryptedOnly(baton.models) &&
            folderAPI.can('create', data) && !virtual && !folderAPI.is('trash', data) &&
            !folderAPI.isExternalFileStorage(data);
    }

    new actionsUtil.Action('oxguard/remencrypt', {
        capabilities: 'guard-drive',
        collection: '!folders',
        matches: function (baton) {
            //all files must be encrypted
            if (!anyOGFiles(baton.data)) return false;
            if (baton.app) {
                return baton.app.folder.getData().then(function (data) {
                    return okToDecrypt(baton, data);
                });
            }
            var folderId = baton.first().folder_id;
            if (folderId) {
                return folderAPI.get(folderId).then(function (data) {
                    return okToDecrypt(baton, data);
                });
            }
            return false;
        },
        action: function (baton) {
            ensureSetup().then(function () {
                auth_core.authorize(baton).then(function (auth) {
                    var warnShares = false;
                    var list = baton.array();
                    for (var file in list) {
                        if (list[file].object_permissions && list[file].object_permissions.length > 0) {
                            warnShares = true;
                        }
                        list[file].file_options = {
                            params: {
                                cryptoAction: 'Decrypt',
                                cryptoAuth: auth
                            }
                        };
                    }
                    if (warnShares) {
                        require(['io.ox/core/tk/dialogs'], function (dialogs) {
                            var dialog = new dialogs.ModalDialog();
                            dialog.text(gt('Shares found'))
                                .append($('<p>').text(gt('You are removing encryption from file(s) that have shares. These shares will no longer work once the file is decrypted.  You can add the shares back once the file is decrypted.')))
                                .append($('<p>').text(gt('Do you wish to proceed?')))
                                .addPrimaryButton('decrypt', gt('Remove encryption'), 'decrypt')
                                .addButton('cancel', gt('Cancel'), 'cancel')
                                .on('decrypt', function () {
                                    doRemEncryptFile(list);
                                })
                                .show();
                        });
                    } else {
                        doRemEncryptFile(list);
                    }
                    baton.openedBy = 'io.ox/files';
                });
            });
        }
    });

    function doRemEncryptFile(list) {
        require(['io.ox/files/api'], function (fileApi) {
            fileApi.copy(list, list[0].folder_id, false)
            .then(function (e) {
                if (e.length === list.length) {
                    for (var i in list) {
                        if (e[i].data) {
                            fileApi.remove([list[i]], true);
                        }
                    }
                } else if (_.isString(e)) {
                    require(['io.ox/core/notifications'], function (notify) {
                        notify.yell('error', e);
                    });
                }
            });
        });
    }

    new actionsUtil.Action('oxguard/open', {
        matches: function (baton) {
            if (baton.data) {
                if (isOGFile(baton.first())) {
                    return (true);
                }
            }
            return (false);
        },
        action: function (baton) {
            auth_core.authorize(baton.data).then(function (auth) {
                var params = {
                    'cryptoAction': 'Decrypt',
                    'cryptoAuth': auth,
                    'session': ox.session
                };
                require(['io.ox/files/api'], function (fileApi) {
                    _(baton.array()).each(function (file) {
                        window.open(fileApi.getUrl(file, 'open', { params: params }));
                    });
                });
                metrics('guard', 'decrypt & open');
            });
        }
    });

    new actionsUtil.Action('oxguard/download', {
        device: '!ios',
        collection: 'some',
        matches: function (baton) {
            var toCheck = baton.models === null ? baton.model : baton.models;
            if (toCheck === undefined) {  // Not a model in versions
                if (baton.data) {
                    return isOGFile(baton.first());
                }
                return false;
            }
            return util.encryptedOnly(toCheck) && util.isNotAttachment(toCheck);
        },
        action: function (baton) {
            var list = baton.array();
            auth_core.authorize(list).then(function (auth) {
                list.map(function (file) {
                    file.filename = (file.filename ? file.filename.replace('.pgp', '') : '');
                    file.params = {
                        'cryptoAction': 'Decrypt',
                        'cryptoAuth': auth,
                        'session': ox.session
                    };
                    file.cache = false;
                });
                actionsUtil.invoke('io.ox/files/actions/downloadversion', baton);
                metrics('guard', 'decrypt & download');
            });
        }
    });

    ////// Main listview handling

    // Call for viewer.  Check if Guard file
    ext.point('io.ox/files/actions/viewer').extend({
        id: 'guardview',
        index: 1,
        collection: 'some && items',
        // matches: hasAnyEncryptedFile,
        matches: function (baton) {
            if (baton.isViewer) return false;
            return hasAnyEncryptedFileInSelection(baton);
        },
        action: function (baton) {
            viewFile(baton);
        }
    });

    ext.point('io.ox/files/actions/viewer/display-version').extend({
        index: 1,
        id: 'guardVersion',
        matches: function (baton) {
            if (hasAnyEncryptedFileInSelection(baton)) {
                baton.stopPropagation();
            }
        }
    });

    // fix for #58841 - Viewer: In a mixture of encrypted and unencrypted files, preview fails if started with an unencrypted file.
    // approach to fulfill user expectation to not being prompted for a password when viewing a single, unencrypted file.
    function hasAnyEncryptedFileInSelection(baton) {

        var selection = baton.array();
        var hasAny = false;

        if (selection.length === 1) {
            // portal provides selection only and no baton.all
            hasAny = isOGFile(selection[0]);
        } else {
            // check selected files for multi selection
            hasAny = selection.some(isOGFile);
        }

        return hasAny;
    }

    // first approach to handle the follow-up story of #58841, after release of Guard 2.10.2 / App Suite 7.10.2
    // another idea would be integrating or tighter coupling ad hoc decrytion into the  Viewer.
    //
    // function hasAnyEncryptedFile(baton) {
    //
    //     var selection = baton.array();
    //     var hasAny = false;
    //
    //     if (selection.length === 1) {
    //         if (baton.all) {
    //             // check all file models in case of a single file selection.
    //             hasAny = baton.all.some(function (model) {
    //                 return isOGFile(model.toJSON());
    //             });
    //         } else {
    //             // portal provides selection only and no baton.all
    //             hasAny = isOGFile(selection[0]);
    //         }
    //     } else {
    //         // check selected files for multi selection
    //         hasAny = selection.some(isOGFile);
    //     }
    //
    //     return hasAny;
    // }

    // Handle viewing Guard file
    function viewFile(baton) {
        // if (hasAnyEncryptedFile(baton)) {
        if (hasAnyEncryptedFileInSelection(baton)) {
            auth_core.authorize(baton).then(function (auth) {
                require(['oxguard/files/downloader'], function (downloader) {
                    downloader.viewFile(baton, 'view', auth);
                });
            });
        }
    }

    ext.point('io.ox/files/dropzone').extend({
        id: 'dragDrop',
        getDropZones: function (baton) {
            if (!capabilities.has('guard-drive')) return;
            var app = baton.app,
                zone2 = new dropzone.Inplace({
                    caption: gt('Drop files here to encrypt')
                });
            zone2.on({
                'show': function () {
                    app.listView.$el.stop().hide();
                },
                'hide': function () {
                    app.listView.$el.fadeIn('fast');
                },
                'drop': function (files) {
                    ensureSetup().then(function () {
                        metrics('guard', 'dropfile upload & encrypt');
                        require(['io.ox/files/upload/main'], function (fileUpload) {
                            fileUpload.setWindowNode(app.getWindowNode());
                            fileUpload.create.offer(files, { folder: app.folder.get(), 'params': { 'cryptoAction': 'Encrypt' } });
                        });
                    });
                }
            });
            return baton.dropZones.push(zone2);
        }
    });

    ////// Functions / util

    function ensureSetup() {

        if (!util.isGuardConfigured()) {
            var def = $.Deferred();
            require(['oxguard/core/createKeys'], function (keys) {
                oxguard.metrics('drive', 'create-keys');
                keys.createKeysWizard()
                .then(def.resolve, def.reject);
            });
            return def;
        }

        return $.when();
    }

    ////////  Encrypting files

    function okToEncrypt(baton, data) {
        var virtual = _.contains(['14', '15'], data.id);
        return baton.collection.has('some', 'modify', 'items') && !util.encryptedOnly(baton.models) &&
            folderAPI.can('create', data) && !virtual && !folderAPI.is('trash', data) &&
            !folderAPI.isExternalFileStorage(data);
    }

    new actionsUtil.Action('oxguard/encrypt', {
        capabilities: 'guard-drive',
        collection: 'some',
        matches: function (baton) {
            //all files must be unencrypted
            if (baton.app) {
                return baton.app.folder.getData().then(function (data) {
                    if (allOGFiles(baton.data)) return false;
                    return okToEncrypt(baton, data);
                });
            }
            var folderId = _.isArray(baton.data) ? baton.data[0].folder_id : baton.data.folder_id;
            if (folderId) {
                return folderAPI.get(folderId).then(function (data) {
                    if (allOGFiles(baton.data)) return false;
                    return okToEncrypt(baton, data);
                });
            }
            return false;
        },
        action: function (baton) {
            var list = _.isArray(baton.data) ? baton.data : [baton.data];
            ensureSetup().then(function () {
                var warnShares = false;
                for (var file in list) {
                    if (list[file].object_permissions && list[file].object_permissions.length > 0) {
                        warnShares = true;
                    }
                    list[file].file_options = {
                        params: {
                            cryptoAction: 'Encrypt'
                        }
                    };
                }
                if (warnShares) {
                    require(['io.ox/core/tk/dialogs'], function (dialogs) {
                        var dialog = new dialogs.ModalDialog();
                        dialog.text(gt('Shares found'))
                            .append($('<p>').text(gt('There are shares associated with the file(s) you are going to encrypt. These shares will no longer work once the file is encrypted.  You can add the shares back once the file is encrypted.')))
                            .append($('<p>').text(gt('Do you wish to proceed?')))
                            .addPrimaryButton('encrypt', gt('Encrypt'), 'encrypt')
                            .addButton('cancel', gt('Cancel'), 'cancel')
                            .on('encrypt', function () {
                                doEncryptFile(list);
                            })
                            .show();
                    });
                } else {
                    doEncryptFile(list);
                }
            });

        }
    });

    function doEncryptFile(list) {
        require(['io.ox/files/api'], function (fileApi) {
            fileApi.copy(list, list[0].folder_id, false)
            .then(function (e) {
                if (e.length === list.length) {
                    for (var i in list) {
                        if (e[i].data) {
                            fileApi.remove([list[i]], true);
                        }
                    }
                } else if (_.isString(e)) {
                    require(['io.ox/core/notifications'], function (notify) {
                        notify.yell('error', e);
                    });
                }
            });
        });
    }

    /////////////////  Util, prompts, etc

    function metrics(target, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: 'drive',
                target: target,
                type: 'click',
                action: action
            });
        });
    }

    /*
    function reload () {
        require(['io.ox/files/main'], function (main) {
            main.getApp().listView.reload();
        });
    }

    */

    // Checks if any of the files are encrypted
    function anyOGFiles(_files) {
        var files = _.isArray(_files) ? _files : [_files];
        var any = false;
        files.map(function (file) {
            any = any || isOGFile(file);
        });
        return any;
    }

    // Checks if all files are Encrypted
    function allOGFiles(_files) {
        var files = _.isArray(_files) ? _files : [_files];
        var all = true;
        files.map(function (file) {
            all = all && isOGFile(file);
        });
        return all;
    }

    function isOGFile(file) {
        try {
            if (!file.filename) return (false);
            if (file.meta !== undefined) {
                if (file.meta.Encrypted) return (true);
            }
            if (/(\.pgp)$/i.test(file.filename)) return true;
        } catch (e) {
            return (false);
        }
        return (false);
    }

    return {
        viewFile: viewFile
    };
});
