/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.file.internal;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import com.openexchange.database.Assignment;
import com.openexchange.database.AssignmentFactory;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.DatabaseAccess;
import com.openexchange.filestore.DatabaseAccessProvider;
import com.openexchange.filestore.utils.AssignmentUsingDatabaseAccess;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.server.ServiceLookup;

/**
 * {@link SproxydDatabaseProvider}
 * Provides the database connection to the guest shards for sproxy prefixes of ext
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.4
 */
public class SproxydDatabaseProvider implements DatabaseAccessProvider {
    
    ServiceLookup services;
    
    public SproxydDatabaseProvider(ServiceLookup services) {
        this.services = services;
    }
    
    /**
     * Pattern match for the file prefix. Should be ext-shardId-userId
     */
    private static final Pattern prefixPattern = Pattern.compile("ext[-_](\\d*)[-_](\\d*)");
    
    /**
     * Get the shard id from the file prefix
     * getGuestShardIdFromPrefix
     *
     * @param directoryPrefix
     * @return
     */
    private static Integer getGuestShardIdFromPrefix(String directoryPrefix) {
        directoryPrefix = Objects.requireNonNull(directoryPrefix, "directoryPrefix must not be null");
        Matcher matcher = prefixPattern.matcher(directoryPrefix);
        if(matcher.find()) {
           return Integer.parseInt(matcher.group(1));
        }
        return null;
    }

    @Override
    public DatabaseAccess getAccessFor(int fileStorageId, String prefix) throws OXException {
        final Integer shardId = getGuestShardIdFromPrefix(prefix);
        if (shardId == null) {  // Not right format
            return null;
        }
        GuardDatabaseService databaseService = services.getServiceSafe(GuardDatabaseService.class);
        String shardName = databaseService.getShardName(shardId);
        AssignmentFactory assignmentFact = services.getServiceSafe(AssignmentFactory.class);
        Assignment assignment = assignmentFact.get(shardName);
        if (assignment == null) {
            throw OXException.general("Unable to get database assignment for Guard shard");
        }
        return new AssignmentUsingDatabaseAccess(assignment, databaseService.getDatabaseService());
        
    }

}
