/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.OGPGPKeyRing;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.PublicExternalKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgpcore.services.PGPKeySigningService;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.keymanagement.responses.PGPSignatureVerificationResultCollectionResponse;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;

/**
 * {@link GetSignaturesAction}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class GetSignaturesAction extends GuardServletAction {

    @FunctionalInterface
    private interface KeyVerificationStrategy {
        Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, long keyId, boolean verifySubkeys) throws OXException;
    }

    private List<KeyVerificationStrategy> verificationStrategies = null;

    /**
     * Initializes a new {@link GetSignaturesAction}.
     *
     */
    public GetSignaturesAction() {

        //Define a list of strategies for searching the provided key (by id) and verify it
        verificationStrategies = Arrays.asList(

            //User's own keys
            new KeyVerificationStrategy() {

                @Override
                public Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, long keyId, boolean verifySubkeys) throws OXException {
                    Collection<PGPSignatureVerificationResult> verificationResults = Collections.emptyList();
                    final GuardKeyService keyService = Services.getService(GuardKeyService.class);
                    final PGPKeySigningService pgpKeySigningService = Services.getService(PGPKeySigningService.class);

                    final GuardKeys key = keyService.getKeysBySubKeyId(verifier.getOXGuardUser().getId(), verifier.getOXGuardUser().getContextId(), keyId);
                    if (key != null) {
                        if (verifySubkeys) {
                            verificationResults = pgpKeySigningService.verifyKey(verifier, key);
                        } else {
                            verificationResults = verifySubKey(pgpKeySigningService, key.getPGPPublicKeyRing(), verifier, keyId);
                        }
                    }
                    return verificationResults;
                }
            },

            //pgp keys of external recipients uploaded by the user
            new KeyVerificationStrategy() {

                @Override
                public Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, long keyId, boolean verifySubkeys) throws OXException {
                    Collection<PGPSignatureVerificationResult> verificationResults = Collections.emptyList();
                    final PGPKeySigningService pgpKeySigningService = Services.getService(PGPKeySigningService.class);
                    PublicExternalKeyService publicExternalKeyService = Services.getService(PublicExternalKeyService.class);
                    OGPGPKeyRing pgpKeyRing = publicExternalKeyService.get(verifier.getOXGuardUser().getId(), verifier.getOXGuardUser().getContextId(), LongUtil.longToHexStringTruncated(keyId));
                    if (pgpKeyRing != null) {
                        if (verifySubkeys) {
                            verificationResults = pgpKeySigningService.verifyKey(verifier, pgpKeyRing.getKeyRing());
                        } else {
                            verificationResults = verifySubKey(pgpKeySigningService, pgpKeyRing.getKeyRing(), verifier, keyId);
                        }
                    }

                    return verificationResults;
                }
            }

        );
    }

    private Collection<PGPSignatureVerificationResult> verifySubKey(PGPKeySigningService pgpKeySigningService, PGPPublicKeyRing pgpKeyRing, UserIdentity verifier, long keyId) throws OXException {
        PGPPublicKey publicKey = pgpKeyRing.getPublicKey(keyId);
        if (publicKey != null) {
            return  pgpKeySigningService.verifyKey(verifier, publicKey);
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND.create();
        }
    }

    private Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, long keyId, boolean verifySubkeys) throws OXException{
        Collection<PGPSignatureVerificationResult> verificationResults = Collections.emptyList();
        for (KeyVerificationStrategy keyVerificationStrategy : verificationStrategies) {
           verificationResults = keyVerificationStrategy.verifyKey(verifier, keyId, verifySubkeys);
           if(verificationResults.size() > 0) {
               break;
           }
        }
        return verificationResults;
    }

    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        final boolean mandatory = true;
        Long keyId = ServletUtils.getLongParameter(request, Parameters.PARAM_KEYID, mandatory);
        boolean verifySubkeys = ServletUtils.getBooleanParameter(request, Parameters.PARAM_SUBKEYS);

        UserIdentity user = getUserIdentityFrom(userSession);

        Collection<PGPSignatureVerificationResult> verificationResults = verifyKey(user, keyId, verifySubkeys);
        if(!verificationResults.isEmpty()) {
            ServletUtils.sendObject(response, new PGPSignatureVerificationResultCollectionResponse(verificationResults, getTranslatorFrom(userSession)));
        }
        else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND.create();
        }
    }
}
