/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import java.util.Objects;
import javax.servlet.http.HttpServletRequest;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link UserIdentityParameterParser}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class UserIdentityParameterParser implements UserIdentityParser {

    private static final String PARAM_PASSWORD = "password";
    private static final String PARAM_E_PASSWORD = "e_password";

    private final HttpServletRequest request;
    private GuardCipherFactoryService cipherFactoryService;
    private GuardMasterKeyService masterKeyService;

    /**
     * Initializes a new {@link UserIdentityParameterParser}.
     * @param request The request to parse the {@link UserIdentity} from
     */
    public UserIdentityParameterParser(GuardMasterKeyService masterKeyService, GuardCipherFactoryService cipherFactoryService, HttpServletRequest request) {
        this.masterKeyService = Objects.requireNonNull(masterKeyService, "masterKeyService must not be null");
        this.cipherFactoryService = Objects.requireNonNull(cipherFactoryService, "cipherFactoryService msut not be null");
        this.request = Objects.requireNonNull(request, "request must not be null");
    }

    private String decryptPassword(String e_password) throws OXException {
        GuardCipherService cipherService = this.cipherFactoryService.getCipherService(GuardCipherAlgorithm.RSA);
        return cipherService.decrypt(e_password, masterKeyService.getMasterKey().getPrivateKey());
    }

    private String parsePassword() throws OXException {
        String password = request.getParameter(PARAM_PASSWORD);
        if(password != null) {
            return password;
        }

        String e_password = request.getParameter(PARAM_E_PASSWORD);
        if(e_password != null) {
            String decryptedPassword = decryptPassword(e_password);
            if(decryptedPassword == null) {
               throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }
            else {
                return decryptedPassword;
            }
        }
        return null;
    }


    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.authentication.UserIdentityParser#parse()
     */
    @Override
    public UserIdentity parse() throws OXException {
        if(request.getParameterMap().size() > 0) {
            String password = parsePassword();
            if(password != null) {
                return new UserIdentity(/*TODO*/null, password);
            }
        }
        return null;
    }
}
