/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.cron;

import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.activity.ActivityTrackingService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEMailMetadataService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEmailMetadata;
import com.openexchange.guard.guest.redirector.GuestRedirectorService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.storage.cache.FileCacheItem;
import com.openexchange.guard.storage.cache.FileCacheStorage;
import com.openexchange.guard.user.OXGuardUser;

/**
 * {@link GuestCleanupTask}
 *
 * Cleanup task for Guests created by Guard.  Cleans the fileCache and MetaData, removing all "shares" for a Guest.
 * This allows periodic Guest cleanup to delete the Guest user
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class GuestCleanupTask implements Runnable {

    private static Logger logger = LoggerFactory.getLogger(GuestCleanupTask.class);
    private final ActivityTrackingService activityTracker;
    private final FileCacheStorage fileCacheStorage;
    private final GuardGuestEMailMetadataService metaDataService;
    private final int daysToKeep;
    private final GuestRedirectorService guestRedirectService;

    public GuestCleanupTask(int daysToKeep, ActivityTrackingService activityTracker, FileCacheStorage fileCacheStorage, GuardGuestEMailMetadataService metaDataService, GuestRedirectorService guestRedirectService) {
        this.daysToKeep = daysToKeep;
        this.activityTracker = activityTracker;
        this.fileCacheStorage = Objects.requireNonNull(fileCacheStorage, "FileCacheStorage service missing");
        this.metaDataService = Objects.requireNonNull(metaDataService, "MetaDataService missing");
        this.guestRedirectService = Objects.requireNonNull(guestRedirectService, "guestRedirectService must not be null");
    }

    private String getDirectoryPrefix(String fullLocation) {
        if(fullLocation.contains(Storage.DELIMITER)) {
            return fullLocation.substring(0, fullLocation.indexOf(Storage.DELIMITER));
        }
        return null;
    }

    /* (non-Javadoc)
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        if (daysToKeep == 0) {
            return;
        }
        if (activityTracker == null) {
            return;
        }
        logger.info("Running Scheduled cleanup of Guests");

        //Getting unused fileCache items
        List<OXGuardUser> users = activityTracker.getExpiredGuests(daysToKeep);
        int failedItems = 0;
        int deletedItems = 0;

        //Remove the unused fileCache items from the storages
        for (OXGuardUser user : users) {
            try {
                List<FileCacheItem> items = fileCacheStorage.findAllForUser(user.getId(), user.getContextId());
                for (FileCacheItem fileCacheItem: items) {
                    //Remove from the file storage
                    Storage storage = Services.getService(Storage.class);
                    storage.deleteEncrObj(fileCacheItem.getLocation());
                    //Remove from the persistent storage (usually the DB)
                    fileCacheStorage.delete(fileCacheItem.getItemId());
                    Integer shardId = storage.getGuestShardIdFromPrefix(getDirectoryPrefix(fileCacheItem.getLocation()));
                    if (shardId != null) {
                        metaDataService.delete(shardId, new GuardGuestEmailMetadata(fileCacheItem.getItemId()));
                    }
                    deletedItems++;
                }
                //remove possible guest redirects for old guard version
                guestRedirectService.deleteRedirect(user.getId(), user.getContextId());
            } catch (OXException ex) {
                logger.error("Error delete object ", ex);
                failedItems++;
            }
        }
        logger.info("Deleted " + deletedItems + " files for " + users.size() + " users");
        if (failedItems > 0) {
            logger.error("Failed deleting " + failedItems + " files");
        }


    }

}
