/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.wksclient.impl;

import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.wksclient.WKSClientService;
import com.openexchange.guard.wksclient.WKSResult;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;

/**
 * {@link WKSClientImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.2
 */
public class WKSClientImpl implements WKSClientService {

    private static int TLS_PORT = 443;

    /**
     * Internal method to extract the host name form the given email address
     *
     * @param email The email
     * @return The host part of the email
     */
    private String getHostFromEmail(String email) {
        return email.substring(email.indexOf("@") + 1);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#findAll(java.lang.String, int)
     */
    @Override
    public Collection<WKSResult> findAll(String email, int timeout) throws OXException {
        return findAll(email, timeout, getHostFromEmail(email), TLS_PORT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#findAll(java.lang.String, int, java.lang.String, int)
     */
    @Override
    public Collection<WKSResult> findAll(String email, int timeout, String host, int port) throws OXException {
        final String verifiedEmail = new EmailValidator().assertInput(email);
        String emailUserHash = CipherUtil.getEmailUserHash(verifiedEmail);
        if (emailUserHash != null && !emailUserHash.isEmpty()) {
            Collection<WKSResult> results = new WKSClient().find(host, emailUserHash, port, timeout);
            //Filter results which do not contains the given email (See WKS-RFC 3.1)
            return results.stream().filter(r -> PGPKeysUtil.containsUID(r.getPublicKeyRing(), verifiedEmail)).collect(Collectors.toList());
        }
        return null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#find(java.lang.String, int)
     */
    @Override
    public Optional<WKSResult> find(String email, int timeout) throws OXException {
        return find(email, timeout, getHostFromEmail(email),TLS_PORT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#find(java.lang.String, int, java.lang.String, int)
     */
    @Override
    public Optional<WKSResult> find(String email, int timeout, String host, int port) throws OXException {
        Collection<WKSResult> results = findAll(email, timeout, host, port);
        //Do not return required or revoked keys and filter keys which do not contain the email (See WKS-RFC 3.1)

        Optional<WKSResult> key = results.stream().filter(r ->
            r.hasValidKey()).findFirst();
        return key;
    }
}
