/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.wksclient.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.impl.client.HttpClientBuilder;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.wksclient.WKSResult;
import com.openexchange.pgp.keys.parsing.PGPPublicKeyRingFactory;

/**
 * {@link WKSClient} Basic WKS clientn for searching and downloading public keys from WKS Servers.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.2
 */
public class WKSClient {

    private static final Logger LOG = LoggerFactory.getLogger(WKSClient.class);

    private static String SCHEMA_PLAIN = "http://";
    private static String SCHEMA_TLS = "https://";
    private static String WELL_KNOWN = "/.well-known/openpgpkey/hu/";
    private static int PORT_PLAIN = 80;
    private static int TLS_PORT = 443;
    private final Mode mode;

    public enum Mode {
        /**
         * HTTP
         */
        PLAIN_TEXT,

        /**
         * HTTPS
         */
        TLS
    }

    /**
     * Initializes a new {@link WKSClient}.
     */
    public WKSClient() {
        this(Mode.TLS);
    }

    /**
     * Initializes a new {@link WKSClient}.
     *
     * @param mode The mode to use.
     */
    WKSClient(Mode mode) {
        this.mode = mode;
    }

    private HttpClient getClient(int connectionTimeout) {
        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectTimeout(connectionTimeout)
            .setSocketTimeout(connectionTimeout)
            .setRedirectsEnabled(false)
            .build();
        return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

    private String buildUrl(String host, String hash, int port) {
        if (host != null && host.endsWith(".")) {
            host = host.substring(0, host.length() - 1);
        }
        return new StringBuilder()
            .append(mode == Mode.TLS ? SCHEMA_TLS : SCHEMA_PLAIN)
            .append(host)
            .append(":")
            .append(Integer.toString(port))
            .append(WELL_KNOWN)
            .append(hash)
            .toString();
    }

    private List<PGPPublicKeyRing> parseKeys(InputStream keyData) throws IOException {
        return PGPPublicKeyRingFactory.createAll(keyData);
    }

    private List<PGPPublicKeyRing> fetchKeys(HttpUriRequest request, int timeout) {
        HttpClient client = getClient(timeout);
        HttpResponse response = null;
        try {
            response = client.execute(request);
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                try (InputStream keyData = response.getEntity().getContent()) {
                    return parseKeys(keyData);
                }
            }
        } catch (IOException e) {
            if (e.getCause() != null && e.getCause().getClass().equals(java.net.SocketTimeoutException.class)) {
                LOG.warn("Timeout waiting for remote key server {} - time remaining in lookup queue was {} ms", request.getURI().getHost(), timeout);
            } else {
                LOG.info("Unable to retrieve keys from server {}", request.getURI().getHost());
                LOG.debug(e.getMessage(), e);
            }
        } finally {
            if (response != null) {
                HttpClientUtils.closeQuietly(response);
            }
            HttpClientUtils.closeQuietly(client);
        }
        return Collections.emptyList();
    }

    private Collection<WKSResult> toWKSResult(Collection<PGPPublicKeyRing> keys) {
        List<WKSResult> ret = new ArrayList<WKSResult>();
        for (PGPPublicKeyRing key : keys) {
            ret.add(new WKSResult(key, WKSKeySources.PUBLIC_WKS));
        }
        return ret;
    }

    /**
     * Performs a WKS key search for the given hash
     *
     * @param host The host
     * @param hash The hash to search the key for
     * @param timeout The connection timeout
     * @return The search results or an empty list in case no key was found for the given hash.
     * @throws IOException
     */
    public Collection<WKSResult> find(String host, String hash, int timeout) {
        return find(host, hash, mode == Mode.TLS ? TLS_PORT : PORT_PLAIN, timeout);
    }

    /**
     * Performs a WKS key search for the given hash
     *
     * @param host The host
     * @param hash The hash to search the key for
     * @param port The port number to use
     * @param timeout The connection timeout
     * @return The search results or an empty list in case no key was found for the given hash.
     * @throws IOException
     */
    public Collection<WKSResult> find(String host, String hash, int port, int timeout) {
        if (timeout <= 0) {
            return Collections.emptyList();
        }

        HttpGet getRequest = new HttpGet(buildUrl(host, hash, port));
        List<PGPPublicKeyRing> foundKeys = fetchKeys(getRequest, timeout);
        return toWKSResult(foundKeys);
    }
}
