/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.wksclient.impl;

import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.CachedKey;
import com.openexchange.guard.keymanagement.storage.RemoteKeyCacheStorage;
import com.openexchange.guard.wksclient.WKSClientService;
import com.openexchange.guard.wksclient.WKSResult;

/**
 * {@link CachingWKSClientImpl} - delegates the WKS search to another {@link WKSClient} but caches the result if trusted.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.2
 */
public class CachingWKSClientImpl implements WKSClientService {

    private static final Logger logger = LoggerFactory.getLogger(CachingWKSClientImpl.class);
    private final WKSClientService delegate;
    private final RemoteKeyCacheStorage remoteKeyCacheStorage;

    /**
     * Initializes a new {@link CachingWKSClientImpl}.
     *
     * @param delegate The {@link WKSClientService} to query if no caching item was found
     */
    public CachingWKSClientImpl(WKSClientService delegate, RemoteKeyCacheStorage remoteKeyCacheStorage) {
        this.delegate = Objects.requireNonNull(delegate, "delegate must not be null");
        this.remoteKeyCacheStorage = Objects.requireNonNull(remoteKeyCacheStorage,"remoteKeyCacheStorage must not be null");
    }

    private CachedKey toCachedKey(WKSResult wksResult) {
        return new CachedKey(wksResult.getPublicKeyRing(), wksResult.getKeySource());
    }

    private WKSResult toWKSResult(CachedKey cachedKey) {
        return new WKSResult(cachedKey.getPublicKeyRing(), cachedKey.getKeySource());
    }

    private Optional<WKSResult> toWKSResult(Optional<CachedKey> cachedKey) {
        return cachedKey.map(ck -> new WKSResult(ck.getPublicKeyRing(), ck.getKeySource()));
    }

    private Collection<WKSResult> toWKSResults(Collection<CachedKey> cachedKeys) {
        return cachedKeys.stream().map( cachedKey -> toWKSResult(cachedKey)).collect(Collectors.toList());
    }

    private boolean writeToCache(WKSResult wksResult) throws OXException {
        if(wksResult.getKeySource() != null && wksResult.getKeySource().isTrusted()) { //Only save from trusted sources
            remoteKeyCacheStorage.insert(toCachedKey(wksResult));
            return true;
        }
        return false;
    }

    private Collection<WKSResult> findAllInCache(String email) throws OXException{
        return toWKSResults(remoteKeyCacheStorage.getByEmail(email));
    }

    private Optional<WKSResult> findFirstInCache(String email) throws OXException {
       return toWKSResult(remoteKeyCacheStorage.getByEmail(email).stream().findFirst());
    }

    private void cache(Collection<WKSResult> results) throws OXException {
        //Add keys from trusted sources to the cache
        if(results != null && !results.isEmpty()) {
            for (WKSResult result : results) {
                if(writeToCache(result)) {
                    logger.debug("Wrote {} found keys from trusted WKS server into cache", results.size());
                }
            }
        }
    }

    private void cache(Optional<WKSResult> result) throws OXException {
        //Add key to cache if trusted
        if (result.isPresent()) {
            if (writeToCache(result.get())) {
                logger.debug("Wrote found key from trusted WKS server into cache");
            }
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#findAll(java.lang.String, int)
     */
    @Override
    public Collection<WKSResult> findAll(String email, int timeout) throws OXException {

        //Search in the cache first
        Collection<WKSResult> results = findAllInCache(email);
        if(results.isEmpty()) {

            //delegate the search if the cache does not contain the key we are looking for
            results = delegate.findAll(email, timeout);
            cache(results);

        }
        else {
            logger.debug("{} Remote keys found in cache", results.size());
        }
        return results;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#find(java.lang.String, int)
     */
    @Override
    public Optional<WKSResult> find(String email, int timeout) throws OXException {

        //Search in the cache first
        Optional<WKSResult> result = findFirstInCache(email);
        if(!result.isPresent()) {
            //delegate the search if the cache does not contain the key we are looking for
            result = delegate.find(email, timeout);
            cache(result);
        }

        return result;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#findAll(java.lang.String, int, java.lang.String, int)
     */
    @Override
    public Collection<WKSResult> findAll(String email, int timeout, String host, int port) throws OXException {

        //Search in the cache first
        Collection<WKSResult> results = findAllInCache(email);
        if(results.isEmpty()) {

            //delegate the search if the cache does not contain the key we are looking for
            results = delegate.findAll(email, timeout, host, port);
            cache(results);
        }
        else {
            logger.debug("{} Remote keys found in cache", results.size());
        }
        return results;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.wksclient.WKSClientService#find(java.lang.String, int, java.lang.String, int)
     */
    @Override
    public Optional<WKSResult> find(String email, int timeout, String host, int port) throws OXException {
        //Search in the cache first
        Optional<WKSResult> result = findFirstInCache(email);
        if(!result.isPresent()) {
            //delegate the search if the cache does not contain the key we are looking for
            result = delegate.find(email, timeout, host, port);
            cache(result);
        }

        return result;
    }
}
