/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services;

import java.util.ArrayList;
import java.util.List;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link PGPSignatureVerificationResultUtil} - A helper class around {@link PGPSignatureVerificationResult}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.2
 */
public class PGPSignatureVerificationResultUtil {

    private static final String VERIFIED_PROPERTY = "verified";
    private static final String DATE_PROPERTY = "signatureCreatedOn";
    private static final String MISSING_PROPERTY = "missing";
    private static final String ERROR_PROPERTY = "error";
    private static final String ISSUER_USERIDS_PROPERTY = "issuerUserIds";
    private static final String ISSUER_KEY_FINGERPRINT_PROPERTY = "issuerKeyFingerprint";
    private static final String ISSUER_KEY_ID_PROPERTY = "issuerKeyId";

    /**
     * Converts the given results into mime-header like String representations
     *
     * @param result The results
     * @return The string representations of the given results
     */
    public static String[] toHeaders (List<PGPSignatureVerificationResult> results) {
        List<String> stringResults = new ArrayList<String>();
        for(PGPSignatureVerificationResult result : results) {
            stringResults.add(toHeader(result));
        }
        String [] res = new String[stringResults.size()];
        return stringResults.toArray(res);
    }

    /**
     * Converts the given result into a mime-header like String representation
     *
     * @param result The result
     * @return The string representation of the given result
     */
    public static String toHeader (PGPSignatureVerificationResult result) {
        StringBuilder sb = new StringBuilder();
        sb.append(VERIFIED_PROPERTY + "=");
        sb.append(result.isVerified() ? "true" : "false");
        sb.append("; " + MISSING_PROPERTY + "=");
        sb.append(result.isMissing() ? "true" : "false");
        if (result.getSignature() != null) {
            sb.append("; " + DATE_PROPERTY + "=");
            sb.append(result.getSignature().getCreationTime().getTime());
        }
        if(result.getIssuerKey() != null) {
            sb.append("; " + ISSUER_KEY_ID_PROPERTY + "=");
            sb.append(result.getIssuerKey().getKeyID());

            sb.append("; " + ISSUER_KEY_FINGERPRINT_PROPERTY + "=");
            sb.append(PGPKeysUtil.getFingerPrint(result.getIssuerKey().getFingerprint()));

        }
        if (result.getIssuerUserIds() != null) {
            sb.append("; " + ISSUER_USERIDS_PROPERTY + "=");
            List<String> userIds = result.getIssuerUserIds();
            for(int i=0; i<userIds.size(); i++) {
                String userId = userIds.get(i);
                sb.append(userId);
                if(i != userIds.size()-1) {
                   sb.append(",");
                }
            }
        }
        if (result.getError() != null) {
            sb.append("; " + ERROR_PROPERTY + "=");
            sb.append(result.getError());
        }
        return Base64.encode(sb.toString());
    }
}
