/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.util.Objects;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.guard.hkpclient.services.HKPClientService;
import com.openexchange.guard.hkpclient.services.RemoteKeyResult;
import com.openexchange.pgp.core.PGPKeyRetrievalStrategy;

/**
 * {@link HKPPGPKeyRetrievalStrategy}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.2
 */
public class HKPPGPKeyRetrievalStrategy implements PGPKeyRetrievalStrategy  {

    private final HKPClientService hkpClient;
    private final String hkpClientToken;
    private final int timeout;
    private final TrustMode trustMode;

    /**
     * {@link TrustMode} - Controlls which key's will be retrieved from the HKP server
     *
     * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
     * @since v7.10.2
     */
    public enum TrustMode{
        /**
         * Retrieval keys from all known HKP sources.
         */
        ALL,

        /**
         * Only retrieval keys from trusted sources.
         */
        TRUSTED_ONLY
    }

    public HKPPGPKeyRetrievalStrategy(HKPClientService hkpClient, int timeout, TrustMode trustMode) {
        this(hkpClient, timeout, null, trustMode);
    }

    public HKPPGPKeyRetrievalStrategy(HKPClientService hkpClient,int timeout, String hkpClientToken, TrustMode trustMode) {
        this.hkpClientToken = hkpClientToken;
        this.timeout = timeout;
        this.trustMode = trustMode;
        this.hkpClient = Objects.requireNonNull(hkpClient, "hkpClient must not be null");
    }

    /* (non-Javadoc)
     * @see com.openexchange.pgp.core.PGPKeyRetrievalStrategy#getSecretKey(long, java.lang.String, char[])
     */
    @Override
    public PGPPrivateKey getSecretKey(long keyId, String userIdentity, char[] password) throws Exception {
        //We cannot fetch private keys using HKP
        return null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.pgp.core.PGPKeyRetrievalStrategy#getPublicKey(long)
     */
    @Override
    public PGPPublicKey getPublicKey(long keyId) throws Exception {
        RemoteKeyResult result = hkpClient.find(hkpClientToken, null, keyId, timeout);
        if(result != null) {
            if (result.getSource().isTrusted() || trustMode == TrustMode.ALL) {
                return result.getRing().getPublicKey(keyId);
            }
        }
        return null;
    }
}
