/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.milter.service.impl;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import com.openexchange.guard.milter.internal.MilterConstants;
import com.openexchange.guard.milter.internal.Responder;
import com.openexchange.guard.milter.service.MilterResponse;

/**
 * {@link ActionResponse}
 *
 * Generates a response to send back to MTA to alter the original email
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class ActionResponse extends MilterResponse {

    private ByteBuffer dataToWrite;
    private int status;

    /* (non-Javadoc)
     * Send the response
     */
    @Override
    public void sendResponse(WritableByteChannel channel) throws IOException {
        Responder.sendPacket(channel, status, (ByteBuffer) dataToWrite.flip());
    }

    /**
     * Reset the response to create new
     */
    public void reset () {
        dataToWrite.clear();
        dataToWrite = null;
        status = 0;
    }

    /**
     * Add recipient to incoming email
     * @param rcp
     * @throws UnsupportedEncodingException
     */
    public void addRcpt(String rcp) throws UnsupportedEncodingException {
        addString(rcp);
        status = MilterConstants.RESPONSE_ADDRCPT;
    }

    /**
     * Remove recipient from incoming email
     * @param rcp
     * @throws UnsupportedEncodingException
     */
    public void delRcpt (String rcp) throws UnsupportedEncodingException {
        addString(rcp);
        status = MilterConstants.RESPONSE_DELRCPT;
    }

    /**
     * Add a header
     * @param header
     * @param value
     * @throws UnsupportedEncodingException
     */
    public void addHeader (String header, String value) throws UnsupportedEncodingException {
        dataToWrite = ByteBuffer.allocate(
            getStringLength(header) +
            getStringLength(value)
            );
        addString(header);
        addString(value);
        status = MilterConstants.RESPONSE_ADDHEADER;
    }

    /**
     * Change a header value.  Will change the first occurence
     * @param header
     * @param value
     * @throws UnsupportedEncodingException
     */
    public void changeHeader(String header, String value) throws UnsupportedEncodingException {
        dataToWrite = ByteBuffer.allocate(4 +
            getStringLength(header) +
            getStringLength(value)
            );
        dataToWrite.putInt(0);  // First location of header
        addString(header);
        addString(value);
        status = MilterConstants.RESPONSE_CHGHEADER;
    }

    /**
     * Replace the body with a new dataBuffer
     * @param dataBuffer
     */
    public void replBody(ByteBuffer dataBuffer) {
        dataToWrite = ByteBuffer.allocate(dataBuffer.limit());
        dataToWrite.put(dataBuffer);
        status = MilterConstants.RESPONSE_REPLBODY;
    }

    private void addString (String data) throws UnsupportedEncodingException {
        if (data != null) {
            byte[] datArray = data.getBytes("UTF-8");
            dataToWrite.put(datArray);
        }
        dataToWrite.put((byte) 0);
    }

    private int getStringLength (String string) {
        if (string == null) return 1;
        int len = string.length() + 1;
        return len;
    }
}
