/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Locale;
import org.bouncycastle.bcpg.CompressionAlgorithmTags;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.sig.Features;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.openpgp.operator.PGPDigestCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPKeyConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.KeyPairSource;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.impl.exceptions.GuardKeyCreationExceptionCodes;
import com.openexchange.guard.update.internal.GuardKeysUpdater;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link KeyCreationServiceImpl} default implementation for {@link KeyCreationService}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyCreationServiceImpl implements KeyCreationService {

    private static final Logger LOG = LoggerFactory.getLogger(KeyCreationServiceImpl.class);
    private static final int[] PREFERRED_SYMETRIC_ALGORITHMS = new int[] { SymmetricKeyAlgorithmTags.AES_256, SymmetricKeyAlgorithmTags.AES_192, SymmetricKeyAlgorithmTags.AES_128 };
    private static int[] PREFERRED_HASH_ALGORITHMS = new int[] { HashAlgorithmTags.SHA512, HashAlgorithmTags.SHA384, HashAlgorithmTags.SHA256, HashAlgorithmTags.SHA224, HashAlgorithmTags.SHA1 };
    private static int[] PREFERRED_COPMRESSION = new int[] { CompressionAlgorithmTags.ZLIB, CompressionAlgorithmTags.BZIP2, CompressionAlgorithmTags.ZIP, CompressionAlgorithmTags.UNCOMPRESSED };
    private final int keyExpirationDays;
    private final int symmetricKeyAlgorithm;
    private final GuardCipherService cipherService;
    private final GuardKeyService keyService;
    private KeyPairSource defaultKeyPairSource;
    private final KeyRecoveryService keyRecoveryService;

    /**
     * Initializes a new {@link KeyCreationServiceImpl}.
     *
     * @param cipherService The {@link GuardCipherService}
     * @param keyService The {@link GuardKeyService}
     * @param keyRecoveryService The {@link KeyRecoveryService}
     * @param keyExpirationDays The amount of days a new Key should be valid
     * @param symmetricKeyAlgorithm The ID of the symmetric encryption algorithm used for password protecting new {@link GuardKeys} instances.
     */
    public KeyCreationServiceImpl(GuardCipherService cipherService,
                                  GuardKeyService keyService,
                                  KeyRecoveryService keyRecoveryService,
                                  int keyExpirationDays,
                                  int symmetricKeyAlgorithm) {

        this.cipherService = cipherService;
        this.keyService = keyService;
        this.keyRecoveryService = keyRecoveryService;
        this.keyExpirationDays = keyExpirationDays;
        this.symmetricKeyAlgorithm = symmetricKeyAlgorithm;
    }

    /**
     * Internal method for creating a recovery token for the new key
     *
     * @param keys The key to create a recovery token for
     * @param password The current password
     * @return The recovery token created for the given key
     * @throws OXException
     */
    private String createKeyRecovery(GuardKeys keys, String password) throws OXException {
        return keyRecoveryService.createRecovery(keys, password);
    }

    private String encryptPrivateKey(PrivateKey privateKey, String password, String salt) throws OXException {
        return cipherService.encrypt(Base64.encode(privateKey.getEncoded()), CipherUtil.getSHA(password, salt), salt);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.KeyCreationService#getKeyPairSource()
     */
    @Override
    public KeyPairSource getKeyPairSource() {
        return this.defaultKeyPairSource;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.KeyCreationService#setKeyPairSource(com.openexchange.guard.keymanagement.services.KeyPairSource)
     */
    @Override
    public void setKeyPairSource(KeyPairSource keyPairSource) {
        this.defaultKeyPairSource = keyPairSource;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.KeyCreationService#create(java.lang.String, java.lang.String, char[])
     */
    @Override
    public GuardKeys create(int userid, int contextid, String name, String email, String password, Locale locale, boolean markAsCurrent, boolean createRecovery) throws OXException {
        return create(this.defaultKeyPairSource, userid, contextid, name, email, password, locale, markAsCurrent, createRecovery);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.KeyCreationService#create(com.openexchange.guard.keymanagement.services.KeyPairCreationStrategy, java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public GuardKeys create(KeyPairSource keyPairSource, int userid, int contextid, String name, String email, String password, Locale locale, boolean markAsCurrent, boolean createRecovery) throws OXException {

        try {
            long start = System.currentTimeMillis();
            // Avoid key name being null
            if (name == null || name.isEmpty()) {
                name = email;
            }

            String salt = CipherUtil.generateSalt();
            char[] pass = CipherUtil.getSHA(password, salt).toCharArray();
            if (password.equals("")) { // If password is blank (unknown), then just create with salt as password
                pass = CipherUtil.getSHA(salt, salt).toCharArray();
                password = salt;
            }

            //Creating raw key pairs for master and for the sub key
            PGPKeyPair masterKey = keyPairSource.get();
            PGPKeyPair subKey = keyPairSource.get();

            if (masterKey == null || subKey == null) {
                throw GuardKeyCreationExceptionCodes.KEY_CREATION_ERROR.create("Unable to retrieve/generate new keys");
            }

            PGPDigestCalculator sha1Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA1);
            PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA256);

            // Encrypting the private key
            PBESecretKeyEncryptor keyEncr = new BcPBESecretKeyEncryptorBuilder(symmetricKeyAlgorithm, sha256Calc).build(pass);

            //Add self signed signature to the master
            PGPSignatureSubpacketGenerator signhashgenGenerator = new PGPSignatureSubpacketGenerator();
            signhashgenGenerator.setKeyFlags(false, KeyFlags.SIGN_DATA | KeyFlags.CERTIFY_OTHER);
            signhashgenGenerator.setPreferredSymmetricAlgorithms(false, PREFERRED_SYMETRIC_ALGORITHMS);
            signhashgenGenerator.setPreferredHashAlgorithms(false, PREFERRED_HASH_ALGORITHMS);
            signhashgenGenerator.setPreferredCompressionAlgorithms(false, PREFERRED_COPMRESSION);
            signhashgenGenerator.setFeature(true, Features.FEATURE_MODIFICATION_DETECTION);

            // Now for subKey (encrypting)
            PGPSignatureSubpacketGenerator enchashgen = new PGPSignatureSubpacketGenerator();
            enchashgen.setKeyFlags(false, KeyFlags.ENCRYPT_COMMS | KeyFlags.ENCRYPT_STORAGE);
            enchashgen.setPreferredSymmetricAlgorithms(false, PREFERRED_SYMETRIC_ALGORITHMS);
            enchashgen.setPreferredHashAlgorithms(false, PREFERRED_HASH_ALGORITHMS);
            enchashgen.setPreferredCompressionAlgorithms(false, PREFERRED_COPMRESSION);
            enchashgen.setFeature(true, Features.FEATURE_MODIFICATION_DETECTION);

            //Set expiration days
            if (keyExpirationDays != 0) {
                int seconds = keyExpirationDays * 24 * 60 * 60;
                signhashgenGenerator.setKeyExpirationTime(false, seconds);
                enchashgen.setKeyExpirationTime(false, seconds);
            }

            //Creating the key generator
            PGPKeyRingGenerator keyRingGen =
                new PGPKeyRingGenerator(PGPSignature.POSITIVE_CERTIFICATION,
                                        masterKey,
                                        name + " <" + email + ">",
                                        sha1Calc,
                                        signhashgenGenerator.generate(),
                                        null,
                                        new BcPGPContentSignerBuilder(masterKey.getPublicKey().getAlgorithm(),HashAlgorithmTags.SHA1),
                                        keyEncr);
            keyRingGen.addSubKey(subKey, enchashgen.generate(), null);

            // Get the private and public encr RSA keys
            JcaPGPKeyConverter conv = new JcaPGPKeyConverter();
            PrivateKey privkey = conv.getPrivateKey(subKey.getPrivateKey());
            PublicKey pubkey = conv.getPublicKey(subKey.getPublicKey());
            String encryptedPrivateKey = encryptPrivateKey(privkey, password, salt);
            // Create key object
            GuardKeys keys = new GuardKeys(keyRingGen, encryptedPrivateKey, pubkey, password, salt);
            keys.setEmail(email);
            keys.setUserid(userid);
            keys.setContextid(contextid);
            keys.setLocale(locale);
            keys.setCurrent(markAsCurrent);
            keys.setRecovery(createRecovery ? createKeyRecovery(keys, password) : "");
            JsonObject version = new JsonObject();
            version.addProperty("version", GuardKeysUpdater.getCurrentVersion());
            keys.setMisc(version.toString());

            LOG.debug("Created PGP/RSA Keys at " + start);
            return keys;
        } catch (PGPException e) {
            throw GuardKeyCreationExceptionCodes.KEY_CREATION_ERROR.create();
        }
    }
}
