/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.guest.impl.comparators;

import java.util.HashMap;
import java.util.Map;
import javax.mail.internet.MimeMessage.RecipientType;
import com.openexchange.guard.guest.MailSortField;

/**
 * {@link ComparatorRegistry} manages a collection of available {@link GuardGuestEmailComparator} instances.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class ComparatorRegistry {

    private static final Map<MailSortField, GuardGuestEmailComparator> COMPARATORS;

    /**
     * A default {@link MailSortField} to use
     */
    public static final MailSortField DEFAULT_SORT_FIELD = MailSortField.SENT_DATE;

    static {
        COMPARATORS = new HashMap<MailSortField, GuardGuestEmailComparator>();
        COMPARATORS.put(MailSortField.RECEIVED_DATE, new ReceivedDateComparator());
        COMPARATORS.put(MailSortField.FROM, new FromComparator());
        COMPARATORS.put(MailSortField.TO, new RecipientComparator(RecipientType.TO));
        COMPARATORS.put(MailSortField.CC, new RecipientComparator(RecipientType.CC));
        COMPARATORS.put(MailSortField.SUBJECT, new SubjectComparator());
        COMPARATORS.put(MailSortField.SIZE, new SizeComparator());
        COMPARATORS.put(MailSortField.SENT_DATE, new SentDateComparator());
        COMPARATORS.put(MailSortField.FLAG_SEEN, new SeenComparator());
        COMPARATORS.put(MailSortField.COLOR_LABEL, new ColorLabelComparator());
    }

    /**
     * Gets a registered comparator for the given {@link MailSortField}
     *
     * @param sortField The {@link MailSortField} to get a comparator for
     * @return The comparator registered for the given sort field, or null if no comparator has been registered for the given sort field.
     */
    public static GuardGuestEmailComparator getComparatorFor(MailSortField sortField) {
        return COMPARATORS.get(sortField);
    }

}
