/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.guest.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Objects;
import com.openexchange.exception.OXException;
import com.openexchange.guard.encryption.EncryptedItem;
import com.openexchange.guard.encryption.EncryptedItemsStorage;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guest.GuestConverterService;
import com.openexchange.guard.guest.exceptions.GuardGuestExceptionCodes;
import com.openexchange.guard.guest.impl.converters.GuestItemConverter;
import com.openexchange.guard.guest.impl.converters.GuestItemConverter.GuestItemConverterResult;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEMailMetadataService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEmailMetadata;
import com.openexchange.guard.guest.redirector.GuestRedirectorService;
import com.openexchange.guard.oxapi.sharing.EmailShareLink;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.storage.cache.FileCacheItem;
import com.openexchange.guard.storage.cache.FileCacheStorage;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link GuestConverterServiceImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class GuestConverterServiceImpl implements GuestConverterService {

    private final EncryptedItemsStorage         encrStorage;
    private final GuestRedirectorService        guestRedirectorService;
    private final GuestItemConverter[]          converters;
    private final Storage                        guestFileStorage;
    private final GuardGuestEMailMetadataService guardGuestEMailMetadataService;
    private final FileCacheStorage              fileCacheStorage;

    private final static String ConvertSuffix = "-rev";

    public GuestConverterServiceImpl(EncryptedItemsStorage encrStorage, GuestRedirectorService guestRedirectorService, Storage guestFileStorage, GuardGuestEMailMetadataService guardMeta, FileCacheStorage fileCacheStorage, GuestItemConverter[] converters) {
        this.encrStorage = Objects.requireNonNull(encrStorage, "EncryptedItemsStorage missing or null");
        this.guestRedirectorService = Objects.requireNonNull(guestRedirectorService, "GuestRedirectorService missing or null");
        this.converters = converters;
        this.guestFileStorage = Objects.requireNonNull(guestFileStorage, "Storage service missing or null");
        this.guardGuestEMailMetadataService = Objects.requireNonNull(guardMeta, "GuardGuestEMailMetadataService missing or null");
        this.fileCacheStorage = Objects.requireNonNull(fileCacheStorage, "FileCacheStorage misisng or null");
    }

    /**
     * Internal method to convert a guest item with a given converter
     * <br>
     * This, can be useful to switch data format between different OX Guard versions.
     *
     * @param guestIdentity The owner of the item to convert.
     * @param objectId The ID of the item to convert.
     * @param newId The new ID of the item once converted.
     * @param converter The converter to use
     * @return GuestItemConverterResult
     * @throws OXException
     */
    private GuestItemConverterResult convert(UserIdentity guestIdentity, String objectId, String newId, GuestItemConverter converter) throws OXException  {
        try (InputStream itemData = this.guestFileStorage.readObj(objectId)) {
            if (itemData != null) {
                ByteArrayOutputStream output = new ByteArrayOutputStream();
                final GuestItemConverterResult result = converter.convert(guestIdentity, itemData, output, objectId);
                if(result == GuestItemConverterResult.CONVERTED) {
                    //Save the converted document
                    this.guestFileStorage.saveEncrObj(guestIdentity.getOXGuardUser().getId(),
                        guestIdentity.getOXGuardUser().getContextId(),
                        newId,
                        output.toByteArray());
                }
                return result;
            }
            return GuestItemConverterResult.NOT_CONVERTED_NOT_NECASSARY;
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Internal method to convert a guest item with a set of converters.
     * <br>
     * This can be useful to switch data format between different OX Guard versions.
     *
     * @param guestUser The owner of the item to convert.
     * @param objectId The ID of the item to convert.
     * @param newId A new ItemId for converted files
     * @throws OXException if there was an error converting the item with one of the given converters.
     */
    private boolean convert(UserIdentity guestUser, String objectId, String newId) throws OXException {
        //Convert the data using the given converters
        boolean converted = false;
        for (GuestItemConverter converter : this.converters) {
            if (convert(guestUser, objectId, newId, converter).equals(GuestItemConverterResult.CONVERTED)) {
                converted = true;
                // It was converted, so future converters should use new ID
                objectId = newId;
            }
        }
        return converted;
    }

    /**
     * Internal method to check whether a conversion of a guest email item is necessary or not.
     *
     * @param guestUser The guest user
     * @param objectId The ID of the item to check
     * @return true, if a conversion is necessary, false otherwise
     * @throws OXException
     */
    private boolean doConvert(UserIdentity guestUser, String objectId) throws OXException {
        GuardGuestEmailMetadata metaData = guardGuestEMailMetadataService.get(
            guestUser.getOXGuardUser().getContextId(),
            guestUser.getOXGuardUser().getId(),
            objectId);
        //We need to convert the item if ..
        return metaData == null || //..no meta data are present
               !metaData.getGuardGuestItemVersion().equals(GuardGuestEmailMetadata.CURRENT_VERSION); //.. or the version does not match
    }

    /**
     * Internal method to create a new ItemId after conversion
     * @param itemId
     * @return
     */
    private String newItemId (String itemId) {
        return itemId + ConvertSuffix;
    }

    /**
     * Internal method to get a list of all items associated with the Guest user and convert
     * @param UserIdentity  -  The userIdentity of the Guest user
     * @return newLinkId -  new ID for the converted file or original itemId
     * @throws OXException
     */
    private String beginConversion (UserIdentity userIdentity, String origLinkId) throws OXException {
        String newLinkId = origLinkId;
        // Get a list of all items in the fileCacheStorage
        List<FileCacheItem> items = fileCacheStorage.findAllForUser(userIdentity.getOXGuardUser().getId(), userIdentity.getOXGuardUser().getContextId());
        for (FileCacheItem item: items) {
            if (doConvert (userIdentity, item.getItemId())) {  // Check if the item requires conversion
                if (convert(userIdentity, item.getItemId(), newItemId(item.getItemId()))) {
                    //  If the original itemId was converted, we'll return the new itemId for creating new link emails
                    if (origLinkId.equals(item.getItemId())) {
                        newLinkId = newItemId(item.getItemId());
                    }
                }
            }
        }
        return newLinkId;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.GuestConverterService#doUpgrade(java.lang.String, int, int)
     */
    @Override
    public String doUpgrade(String itemId, UserIdentity userIdentity, String language) throws OXException {
        String url = guestRedirectorService.getRedirect(userIdentity.getOXGuardUser().getId(), userIdentity.getOXGuardUser().getContextId());
        if (url != null) {
            return url;
        }
        ///// Do upgrade
        if (this.converters == null || this.converters.length == 0) {
            throw GuardGuestExceptionCodes.MISSING_REQUIRED_CONVERTERS.create();
        }
        // Get the original senders context by pulling item from encrypted objects table
        EncryptedItem item = encrStorage.getById(itemId.replace("pgp-", ""), userIdentity.getOXGuardUser().getId(), userIdentity.getOXGuardUser().getContextId());
        if (item == null) {
            throw GuardGuestExceptionCodes.ITEM_NOT_FOUND.create();
        }
        int context = item.getOwnerCid();
        int userId = item.getOwnerId();

        // Do conversion
        String newItemId = beginConversion(userIdentity, itemId);

        // Create redirect with new ID
        String redirect = new EmailShareLink().createEmailShare(userId, context, userIdentity.getIdentity(), newItemId, language);
        if (redirect == null) {
            throw GuardGuestExceptionCodes.PROBLEM_WITH_CONVERSION.create();
        }

        // Add the url redirect to the redirect table for future reference
        guestRedirectorService.addRedirect(userIdentity.getOXGuardUser().getId(),
            userIdentity.getOXGuardUser().getContextId(),
            redirect);
        return redirect;
    }

}
