/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.common.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import com.j256.simplemagic.ContentInfo;
import com.j256.simplemagic.ContentInfoUtil;

/**
 * {@link ContentTypeDetectionUtil} is a helper class for guessing the content-type of binary data.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class ContentTypeDetectionUtil {

    public final static int READ_SIZE = 10 * 1024;

    /**
     * Internal implementation of content-type guessing.
     *
     * @param data The data to guess the content-type for
     * @return The guessed content-type for the data, or null if it could not be guessed.
     */
    private static String getContentType(byte[] data) {
        final ContentInfoUtil util = new ContentInfoUtil();
        final ContentInfo contentInfo = util.findMatch(data);
        return contentInfo != null ? contentInfo.getMimeType() : null;
    }

    /**
     * Guesses the content-type of an {@link InputStream}.
     *
     * The inputStream must support the <code>mark</code> and <code>reset</code> methods.
     *
     * @param inputStream The {@link InputStream} to guess the content-type for.
     * @return The guessed content-type, or null if no bytes could be read from the stream
     * @throws UnsupportedOperationException if the given inputStream does not support <code>mark</code> and <code>reset</code> methods.
     * @throws IOException
     */
    public static String guessContentType(InputStream inputStream) throws IOException {

        if (!inputStream.markSupported()) {
            throw new UnsupportedOperationException("The input stream does not support mark and reset methods.");
        }
        byte[] bytes = new byte[READ_SIZE];
        inputStream.mark(READ_SIZE);
        final int numRead = inputStream.read(bytes);
        if (numRead < 0) {
            return null;
        }
        if (numRead < bytes.length) {
            // move the bytes into a smaller array
            bytes = Arrays.copyOf(bytes, numRead);
        }

        final String contentType = getContentType(bytes);
        inputStream.reset();
        return contentType;
    }
}
