/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.autocrypt.database;

import java.io.IOException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import com.openexchange.exception.OXException;
import com.openexchange.guard.autocrypt.impl.AutoCryptUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * {@link AutoCryptKey}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.2
 */
public class AutoCryptKey {

    public enum Preference {
        mutual ("mutual"),
        none ("none");

        private String name;

        private Preference(String name) {
            this.name = name;
        }

        public boolean equalsName(String toCheck) {
            return this.name.equals(toCheck);
        }

        public String toString() {
            return this.name();
        }

    }

    private PGPPublicKeyRing keyRing;
    private int userId;
    private int cid;
    private String email;
    private long lastUpdate;
    private String hash;
    private Preference preference;
    private boolean verified;

    public AutoCryptKey (int userId, int cid, String email, PGPPublicKeyRing ring, long lastUpdate, Preference preference) {
        this.userId = userId;
        this.cid = cid;
        this.email = email;
        this.keyRing = ring;
        this.lastUpdate = lastUpdate;
        this.preference = preference;
        this.hash = AutoCryptUtils.getHash(this);
    }

    public AutoCryptKey (int userId, int cid, String email, byte[] ringBytes, long lastUpdate, String preference, String hash, boolean verified) throws OXException {
        this.userId = userId;
        this.cid = cid;
        this.email = email;
        try {
            this.keyRing = new PGPPublicKeyRing(ringBytes, new BcKeyFingerprintCalculator());
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create("Unable to create Autocrypt Key");
        }
        this.lastUpdate = lastUpdate;
        this.preference = preference.equals("mutual") ? Preference.mutual : Preference.none;
        this.hash = hash;
        this.verified = verified;
    }

    public boolean equals(AutoCryptKey otherKey) {
        if (this.hash == null) return false;
        return this.hash.equals(otherKey.getHash());
    }

    public int getUserId() {
        return userId;
    }

    public int getCid() {
        return cid;
    }

    public String getEmail () {
        return email;
    }

    public PGPPublicKeyRing getPGPPublicKeyRing () {
        return keyRing;
    }

    public long getLastUpdate() {
        return lastUpdate;
    }

    public Preference getPreference() {
        return preference;
    }

    public String getPreferenceString () {
        return preference.toString();
    }

    public String getHash() {
        return this.hash;
    }

    public boolean getVerified() {
        return this.verified;
    }

    public void setVerified(boolean verified) {
        this.verified = verified;
    }

}
