/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.autocrypt;

import java.util.ArrayList;
import javax.mail.internet.MimeMessage;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.autocrypt.impl.AutoCryptKeysResponse;
import com.openexchange.guard.autocrypt.impl.AutoCryptStartInfoResponse;
import com.openexchange.guard.autocrypt.impl.TransferResponse;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;

/**
 * {@link AutoCryptService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.2
 */
public interface AutoCryptService {

    /**
     * Get AutoCrypt string for outgoing email
     * @param userId
     * @param cid
     * @param emailAddress, should be the from address
     * @return
     * @throws OXException
     */
    public String getOutgoingHeaderString (int userId, int cid, String emailAddress) throws OXException;

    /**
     * Import a key from a header string
     * @param userId
     * @param cid
     * @param header
     * @param email From addresss
     * @param date  Sent time in ticks
     * @param add  If the key should be added automatically
     * @param confiremd If the key was confirmed by the user
     * @return
     * @throws OXException
     */
    public AutoCryptKeysResponse importKey(int userId, int cid, String header, long date, boolean add, boolean confirmed) throws OXException;

    /**
     * Check for recipient key
     * @param userId  userId of sender
     * @param cid  cid of sender
     * @param email  email address of recipient
     * @return
     * @throws OXException
     */
    public RecipKey getRecipKey(int userId, int cid, String email) throws OXException;

    /**
     * Adds gossip headers for encrypted outgoing emails
     * @param userId
     * @param cid
     * @param recips
     * @return
     */
    public MimeMessage addGossipHeaders(MimeMessage msg, int userId, int cid, ArrayList<RecipKey> recips) throws OXException;

    /**
     * Get password information from an autocrypt start email
     * @param startAttachment
     * @return Object containing information regarding password length, as well as start characters if any
     * @throws OXException
     */
    public AutoCryptStartInfoResponse getStartData(String startAttachment) throws OXException;

    /**
     * Add the pgp secret key from a autocrypt start email
     * @param attachment
     * @param key
     * @return
     * @throws OXException
     */
    public PGPSecretKeyRing getKeyFromAttachment(String attachment, String key) throws OXException;

    /**
     * Send a private key transfer email to self, returning response with passcode
     * @param key to be transferred
     * @param password
     * @param senderIp IP address of the user
     * @return
     * @throws OXException
     */
    public TransferResponse sendTransferEmail(GuardKeys key, String password, String host, String senderIp) throws OXException;


}
