/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.KeyImportService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.guard.servlets.keymanagement.responses.GuardKeyInfoCollectionResponse;
import com.openexchange.pgp.keys.parsing.KeyRingParserResult;
import com.openexchange.pgp.keys.parsing.PGPKeyRingParser;

/**
 * {@link UploadKeysAction} allows to upload custom, client created, public and private keys.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class UploadKeysAction extends GuardServletAction {

    private static final int MAX_KEY_FILE_UPLOAD_SIZE = 1000000; /* 1 MB */
    private static final String KEY_PARAMETER_NAME= "key";
    private static final String KEY_PASSWORD_PARAMETER_NAME = "keyPassword";
    private static final String NEW_PASSWORD_PARAMETER_NAME = "newPassword";

    /**
     * Internal method to remove duplicated key rings
     *
     * @param importedKeyRings The list of keys rings
     * @return The list where duplicates have been removed
     */
    private Collection<GuardKeys> removeDuplicatedImportedKeyResults(Collection<GuardKeys> importedKeyRings) {

        Collection<GuardKeys> ret = new ArrayList<GuardKeys>(importedKeyRings.size());
        Collection<Long> recognizedIds = new ArrayList<Long>(importedKeyRings.size());

        importedKeyRings.forEach(key -> {
            if(!recognizedIds.contains(new Long(key.getPGPPublicKey().getKeyID()))){
                recognizedIds.add(new Long(key.getPGPPublicKey().getKeyID()));
                ret.add(key);
            }
        });

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {

        //Parsing
        FileUploadHandler fileUploadHandler = new FileUploadHandler(Services.getService(GuardConfigurationService.class));
        List<PGPSecretKeyRing> parsedSecretKeyRings = null;
        List<PGPPublicKeyRing> parsedPublicKeyRings = null;
        Collection<FileItem> items = fileUploadHandler.parseItems(request, MAX_KEY_FILE_UPLOAD_SIZE);
        String keyPassword = null;
        String newPassword = null;
        PGPKeyRingParser keyParser = Services.getService(PGPKeyRingParser.class);
        if (items.size() > 0) {
            //Get actual Key data
            for (FileItem item : items) {
                if(item.getFieldName().equals(KEY_PASSWORD_PARAMETER_NAME)) {
                    keyPassword = item.getString(StandardCharsets.UTF_8.displayName());
                }
                else if(item.getFieldName().equals(NEW_PASSWORD_PARAMETER_NAME)) {
                    newPassword = item.getString(StandardCharsets.UTF_8.displayName());
                }
                else{
                    KeyRingParserResult parseResult = keyParser.parse(item.getInputStream());
                    if(parseResult != null) {
                        parsedSecretKeyRings = new ArrayList<PGPSecretKeyRing>();
                        parsedSecretKeyRings.addAll(parseResult.toSecretKeyRings());
                        parsedPublicKeyRings = new ArrayList<PGPPublicKeyRing>();
                        parsedPublicKeyRings.addAll(parseResult.toPublicKeyRings());
                    }
                }
            }
        } else {
            throw GuardCoreExceptionCodes.MULTIPART_UPLOAD_MISSING.create();
        }
        if((parsedPublicKeyRings == null || parsedPublicKeyRings.isEmpty()) &&
           (parsedSecretKeyRings == null || parsedSecretKeyRings.isEmpty()) ) {
            throw GuardCoreExceptionCodes.INVALID_PARAMETER_VALUE.create(KEY_PARAMETER_NAME);
        }
        final String email = new EmailValidator().assertInput(new Api(new OxCookie(request.getCookies()),request).getPrimary(), "email");
        //Check if password parameters are present when importing secret keys
        if(parsedSecretKeyRings.size() > 0) {
            if(keyPassword == null || keyPassword.isEmpty()) {
                parsedSecretKeyRings.clear();
            }
            if(!parsedSecretKeyRings.isEmpty() && (newPassword == null || newPassword.isEmpty())) {
                throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(NEW_PASSWORD_PARAMETER_NAME);
            }
        }

        //Import parsed keys
        KeyImportService keyImportService = Services.getService(KeyImportService.class);
        Collection<GuardKeys> importedKeyRings = keyImportService.importPrivateKeyRing(userSession.getGuardUserId(),
            userSession.getGuardContextId(),
            email,
            userSession.getLocale(),
            keyPassword,
            newPassword,
            parsedSecretKeyRings.toArray(new PGPSecretKeyRing[parsedSecretKeyRings.size()]));
        importedKeyRings.addAll(
            keyImportService.importPublicKeyRing(userSession.getGuardUserId(),
            userSession.getGuardContextId(),
            email,
            userSession.getLocale(),
            parsedPublicKeyRings.toArray(new PGPPublicKeyRing[parsedPublicKeyRings.size()])));

        //the result list can contain duplicates because if public + private keys are imported / merged keys are also returned.
        //Cleaning up for a client.
        importedKeyRings = removeDuplicatedImportedKeyResults(importedKeyRings);
        ServletUtils.sendObject(response, new GuardKeyInfoCollectionResponse(newPassword, importedKeyRings));
    }

}
