/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.guest;

import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.UserIdentityPasswordlessGuestParser;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link GuardGuestServletAction}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public abstract class GuardGuestServletAction extends GuardServletAction {

    /**
     * Parses a {@link UserIdentity} for a guest from the given {@link JSONObject} with just containing the email and no password
     *
     * @param json the json object to parse the guest identity from
     * @return The parsed {@link UserIdentity} with just the email set and no password, or null if no such guest user exists
     * @throws OXException
     */
    protected UserIdentity getPasswordLessGuestUserIdentityFrom(JSONObject json) throws OXException {
        return new UserIdentityPasswordlessGuestParser(Services.getService(OXUserService.class), json).parse();
    }

    /**
     * Parses a {@link UserIdentity} for a guest from the given {@link JSONObject} just containing the email and no password,
     * or throws an exception if the request does not contain enough information to parse the guest, or the guest does not exists.
     *
     * @param json the json object to parse the guest identity from
     * @return The parsed {@link UserIdentity} with just the email set and no password.
     * @throws OXException
     */
    protected UserIdentity requirePasswordLessGuestUserIdentityFrom(JSONObject json) throws OXException {
        UserIdentity userIdentity = new UserIdentityPasswordlessGuestParser(Services.getService(OXUserService.class), json).parse();
        final boolean require = true;
        return requireOrReturn(userIdentity, require);
    }

    /**
     * Tries to get a valid guest {@link UserIdentity} from the given json object
     * <br>
     * This tries to return a {@link UserIdentity} for a valid guard authentication token provided in the given json.
     * <br>
     * If the json does not contain a valid authentication token then this method will try to determine a password-less guest by email
     * <br>
     *
     * @param json The json to parse the identity from
     * @return The identity for an auth-token provided in the json, or a passwordless guest identity for a given email, or null if no identity could be obtained.
     * @throws OXException
     * @throws JSONException
     */
    protected UserIdentity getAnyGuestIdentityFrom(JSONObject json) throws OXException, JSONException {
        UserIdentity guestIdentity = null;
        if (json.has("auth")) {
            guestIdentity = getUserIdentityFrom(json.getJSONObject("auth"));
            if (guestIdentity == null) {
                guestIdentity = getPasswordLessGuestUserIdentityFrom(json.getJSONObject("auth"));
            }
        }
        return guestIdentity;
    }

    /**
     * Tries to get a valid guest {@link UserIdentity} from the given json object
     * <br>
     * This tries to return a {@link UserIdentity} for a valid guard authentication token provided in the given json.
     * <br>
     * If the json does not contain a valid authentication token then this method will try to determine a a password-less guest by email, or throw an exception if no identity could be obtained.
     * <br>
     *
     * @param json The json to parse the identity from
     * @return The identity for an auth-token provided in the json, or a passwordless guest identity for a given email, or null if no no identity could be obtained.
     * @throws OXException
     * @throws JSONException
     */
    protected UserIdentity requireAnyGuestIdentityFrom(JSONObject json) throws OXException, JSONException {
        UserIdentity anyGuestIdentityFrom = getAnyGuestIdentityFrom(json);
        final boolean require = true;
        return requireOrReturn(anyGuestIdentityFrom, require);
    }
}
