/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.fileupload;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;

/**
 * {@link JsonFileUploadHandler} adds various JSON handling to a {@link FileUploadHandler}.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class JsonFileUploadHandler {

    private final FileUploadHandler fileUploadFactory;

    /**
     * Initializes a new {@link JsonFileUploadHandler}.
     *
     * @param fileUploadHandler The {@link FileUploadHandler} to add the JSON handling for.
     */
    public JsonFileUploadHandler(FileUploadHandler fileUploadHandler) {
        this.fileUploadFactory = fileUploadHandler;
    }

    /**
     * Parses the JSON part from the incoming input stream.
     *
     * @param jsonInputStream the input stream to parse the JSON from
     * @return The json object parse from the input stream
     * @throws IOException
     * @throws JSONException
     */
    public JSONObject getJsonFrom(InputStream jsonInputStream) throws IOException, JSONException {
        ByteArrayOutputStream jsonData = new ByteArrayOutputStream();
        IOUtils.copy(jsonInputStream, jsonData);
        return new JSONObject(new String(jsonData.toByteArray(), StandardCharsets.UTF_8));
    }

    /**
     * Gets the JSON object from a parsed list of fileItems.
     *
     * @param fileItems The parsed list of FileItems
     * @param name The name of the JSON item
     * @return The JSON object, or null if the list does not contain a JSON object
     * @throws IOException
     * @throws JSONException
     * @throws OXException If mandatory is set to true and the given FileItem was not found.
     */
    public JSONObject getJsonFrom(Collection<FileItem> fileItems, String name) throws IOException, JSONException {
        try {
            return getJsonFrom(fileItems, name, false);
        } catch (OXException e) {
            return null;
        }
    }

    /**
     * Gets the JSON object from a parsed list of fileItems.
     *
     * @param fileItems The parsed list of FileItems
     * @param name The name of the JSON item
     * @param mandatory True to throw an OXException if the JSON is not present in the given fileItems, False to return null.
     * @return The JSON object, or null if the list does not contain a JSON object and mandatory is set to false
     * @throws IOException
     * @throws OXException If mandatory is set to true and the given FileItem was not found.
     * @throws JSONException
     */
    public JSONObject getJsonFrom(Collection<FileItem> fileItems, String name, boolean mandatory) throws IOException, OXException, JSONException {
        try(InputStream jsonStream = fileUploadFactory.getFileItemStreamFrom(fileItems, name, mandatory);){
            if (jsonStream != null) {
                return getJsonFrom(jsonStream);
            }
            return null;
        }
    }
}
