/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.demo;

import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.demo.exceptions.DemoExceptionCodes;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.support.UserDeleter;

/**
 * This class provides functionalities which are useful for running OX Guard in
 * a development or demonstration environment; For accessing the class it is required
 * to set com.openexchange.guard.demo to true.
 *
 * @author @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 */
public class DemoUtil {

    private static final Logger LOG = LoggerFactory.getLogger(DemoUtil.class);

    /**
     * Throws an OXException if OX Guard is not in demo mode
     *
     * @throws OXException
     */
    protected void assertDemoMode() throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        if (!configService.getBooleanProperty(GuardProperty.demo)) {
            LOG.error("Attempted access demo functionality, but not configured for demo.  Add demo:true to config file is needed");
            throw DemoExceptionCodes.DEMO_MODE_DISABLED_ERROR.create();
        }
    }

    /**
     * Internal method to delete a user's Guard key by email
     *
     * @param email The emai object of the user
     * @throws OXException
     */
    private void deleteByEmail(Email email) throws OXException {
        if (email != null) {
            int userId = email.getUserId();
            int contextId = email.getContextId();
            new UserDeleter().delete(userId, contextId);
        } else {
            throw DemoExceptionCodes.UNKNOWN_EMAIL_ERROR.create(email);
        }
    }

    /**
     * Deletes all known OX Guard keys  WITHOUT creating any backups
     * (com.openexchange.guard.demo has to be "true")
     * @throws OXException
     */
    public int deleteAllUserKeys() throws OXException {
        assertDemoMode();

        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        List<Email> allEmails = ogEmailStorage.getEmails();
        int count = 0;
        for(Email email : allEmails) {
            deleteByEmail(email);
            count++;
        }
        return count;
    }

    /**
     * Deletes all known OX Guard keys for a given email WITHOUT creating any backups
     * (com.openexchange.guard.demo has to be "true")
     *
     * @param email The email of the user to delete all keys for
     * @throws OXException
     */
    public void deleteUserKeys(String email) throws OXException {
        //ensure OX Guard is running in demo mode; or throw an exception
        assertDemoMode();

        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        Email emailObject = ogEmailStorage.getByEmail(email);
        if(emailObject != null) {
            deleteByEmail(emailObject);
        }
        else {
            throw DemoExceptionCodes.UNKNOWN_EMAIL_ERROR.create(email);
        }
    }

    /**
     * Resets the Anti-Abuse entry for a given user
     *
     * @param contextId The context ID
     * @param userId The user ID
     * @throws OXException
     */
    public void resetBad(int contextId, int userId) throws OXException {

        assertDemoMode();

        GuardAntiAbuseService antiAbuseService = Services.getService(GuardAntiAbuseService.class);
        final String username = userId + "-" + contextId;
        antiAbuseService.removeBad(username);
    }
}
