/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.user;

import com.openexchange.pgp.keys.common.PGPSymmetricKey;

/**
 * {@link UserIdentity} represents a generic PGP user.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class UserIdentity {

    private final char[]    password;
    private String          identity;
    private PGPSymmetricKey symmetricKey;
    private OXUser          oxUser;
    private OXGuardUser     oxGuardUser;

    /**
     * Initializes a new {@link UserIdentity}.
     *
     * @param identity The identity
     * @param password The password
     */
    public UserIdentity(String identity, String password) {
        this.identity = identity;
        this.password = password != null ? password.toCharArray() : null;
    }

    /**
     * Initializes a new {@link UserIdentity}.
     *
     * @param identity the identity
     */
    public UserIdentity(String identity) {
        this.identity = identity;
        this.password = new char[] {};
    }

    /**
     * Gets the identity
     *
     * @return The identity
     */
    public String getIdentity() {
        return identity;
    }

    /**
     * Sets the identity
     *
     * @param identity The identity to set
     * @return this
     */
    public UserIdentity setIdentity(String identity) {
        this.identity = identity;
        return this;
    }

    /**
     * Gets the password
     *
     * @return The password
     */
    public char[] getPassword() {
        return password;
    }

    /**
     * Gets the user's symmetric PGP session key which can be used alternative to the password for decryptingn PGP data.
     *
     * @return A PGP symmetric session key.
     */
    public PGPSymmetricKey getPGPSessionKey() {
        return symmetricKey;
    }

    /**
     * Sets a symmetric PGP session key to use for decryption.
     *
     * @param pgpSessionKey A symmetric session key used to decryption.
     * @return this
     */
    public UserIdentity setPGPSession(PGPSymmetricKey symmetricKey) {
        this.symmetricKey = symmetricKey;
        return this;
    }

    /**
     * Gets the related OX user for this PGP user entity, or null if this entity is not related to an regular OX Appsuite account.
     *
     * @return The OX Appsuite user, or null, if this entity is not related to a regular OX Appsuite account.
     */
    public OXUser getOXUser() {
        return this.oxUser;
    }

    /**
     * Sets the OX Appsuite user related to this PGP entity.
     *
     * @param oxUser the related OX User
     * @return this, for a fluent like method call
     */
    public UserIdentity setOXUser(OXUser oxUser) {
        this.oxUser = oxUser;
        return this;
    }

    /**
     * Returns if this PGP entity is related to an OX account or not.
     *
     * @return True if this PGP entity is related to an OX account, false otherwise.
     */
    public boolean isOXUser() {
        return this.oxUser != null;
    }

    /**
     * Gets the related OX Guard user information for this PGP user entity, or null if this entity is not related to an OX Guard account.
     *
     * @return The OX Guard user, or null, if this entity is not related to an OX Guard account.
     */
    public OXGuardUser getOXGuardUser() {
        return this.oxGuardUser;
    }

    /**
     * Sets the OX Guard user related to this PGP entity.
     *
     * @param oxGuardUser The related OX Guard user
     * @return This, for a fluent like method call
     */
    public UserIdentity setOXGuardUser(OXGuardUser oxGuardUser) {
        this.oxGuardUser = oxGuardUser;
        return this;
    }

    /**
     * Returns if this PGP entity is related to an, non guest, OX Guard user.
     *
     * @return True if the PGP entity is related to a regular OX Guard account which is not a guest.
     */
    public boolean isOXGuardUser() {
        return this.oxGuardUser != null && !this.oxGuardUser.isGuest();
    }

    /**
     * Returns if this PGP entity is related to an OX Guard Guest Account or not.
     * @return True, if the PGP entity is related to an OX Guard Guest account, false otherwise
     */
    public boolean isOXGuardGuest() {
       return this.oxGuardUser != null && this.oxGuardUser.isGuest();
    }
}
