/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage.cache;

import java.io.Serializable;
import java.util.List;
import com.openexchange.exception.OXException;
import com.openexchange.guard.caching.GenericCache;
import com.openexchange.guard.caching.GenericCacheFactory;

/**
 * {@link CachingFileCacheStorage}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class CachingFileCacheStorage implements FileCacheStorage {

    private final FileCacheStorage delegate;
    private GenericCache<FileCacheItem> cache;

    /**
     * Initializes a new {@link CachingFileCacheStorage}.
     *
     * @param cacheFactory The factory for creating the cache to use
     * @param delegate The delegate to enhance with caching
     */
    public CachingFileCacheStorage(GenericCacheFactory cacheFactory, FileCacheStorage delegate) {
        this.cache = cacheFactory.createCache("GuestFileItem");
        this.delegate = delegate;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#getById(java.lang.String)
     */
    @Override
    public FileCacheItem getById(String itemId) throws OXException {
        Serializable key = cache.createKey(itemId);
        FileCacheItem item = this.cache.get(key);
        if(item == null) {
            item = delegate.getById(itemId);
            if(item != null) {
                cache.put(key, item);
            }
        }
        return item;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#getByIdForUser(java.lang.String, int)
     */
    @Override
    public FileCacheItem getByIdForUser(String itemId, int userId) throws OXException {
        Serializable key = cache.createKey(itemId);
        FileCacheItem item = this.cache.get(key);
        if(item == null) {
            item = delegate.getByIdForUser(itemId, userId);
            if(item != null) {
                cache.put(key, item);
            }
        }
        return item;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#getForUser(int, int, int)
     */
    @Override
    public List<FileCacheItem> getForUser(int userId, int offset, int maxCount) throws OXException {
        return delegate.getForUser(userId, offset, maxCount);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#insert(java.lang.String, int, java.lang.String)
     */
    @Override
    public void insert(String itemId, int userId, String path) throws OXException {
        delegate.insert(itemId, userId, path);
        cache.remove(cache.createKey(itemId));
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#updateLastDate(java.lang.String)
     */
    @Override
    public void updateLastDate(String itemId) throws OXException {
        delegate.updateLastDate(itemId);
        cache.remove(cache.createKey(itemId));
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#delete(java.lang.String)
     */
    @Override
    public void delete(String itemId) throws OXException {
        delegate.delete(itemId);
        cache.remove(cache.createKey(itemId));
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#findOld(int)
     */
    @Override
    public List<FileCacheItem> findOld(int days) throws OXException {
        return delegate.findOld(days);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.cache.FileCacheStorage#findAllForUser(int)
     */
    @Override
    public List<FileCacheItem> findAllForUser(int userId) throws OXException {
        return delegate.findAllForUser(userId);
    }
}
