/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.sharing.impl;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import org.json.JSONException;
import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.ldap.User;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.AccountCreationService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.sharing.SharingService;
import com.openexchange.guard.sharing.exceptions.GuardSharingExceptionCodes;
import com.openexchange.guard.sharing.osgi.Services;
import com.openexchange.guard.user.OXUser;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.packethandling.PacketProcessor;
import com.openexchange.pgp.core.packethandling.RelaceAllRecipientsPacketProcessorHandler;
import com.openexchange.pgp.core.packethandling.SecretKeyForPacketService;
import com.openexchange.user.UserService;

/**
 * {@link SharingServiceImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class SharingServiceImpl implements SharingService {

    /* (non-Javadoc)
     * @see com.openexchange.guard.sharing.SharingService#shareFile(java.io.InputStream, org.json.JSONObject, com.openexchange.guard.user.UserIdentity, java.io.OutputStream)
     */
    @Override
    public void shareFile(InputStream origData, List<Integer> userIds, UserIdentity userIdentity, OutputStream out) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        // Pull current key here for re-encryption
        GuardKeys owner = keyService.getKeys(userIdentity.getOXUser().getId(), userIdentity.getOXUser().getContextId());
        if (owner == null) {
            throw GuardSharingExceptionCodes.OWNER_KEY_NOT_FOUND.create();
        }
        SecretKeyForPacketService keyPacketService = new SecretKeyForPacketImpl(userIdentity);
        try {
            List<PGPPublicKey> keys = getPublicKeys(owner, userIds, keyService);
            PacketProcessor processor = new PacketProcessor();
            processor.process(
                origData,
                out,
                new RelaceAllRecipientsPacketProcessorHandler(keyPacketService, keys.toArray(new PGPPublicKey[keys.size()])),
                false);
        } catch (Exception e) {
            throw GuardSharingExceptionCodes.PROBLEM_UPDATING_ENCRYPTION.create();
        }

    }

    /**
     * Get a list of public keys for a share.  Create keys if needed.
     * @param ownerKey The owner of the file.  Will be added to the list
     * @param userIds User Ids of the shares.  Will all have the same context as the owner
     * @param keyService  Guard Key Service
     * @return new list of PGPPublic keys
     * @throws JSONException
     * @throws OXException
     */
    private List<PGPPublicKey> getPublicKeys (GuardKeys ownerKey, List<Integer> userIds, GuardKeyService keyService) throws JSONException, OXException {
        ArrayList<PGPPublicKey> keys = new ArrayList<PGPPublicKey>();
        keys.add(new RecipKey(ownerKey).getEncryptionKey());
        UserService userService = Services.getService(UserService.class);
        for (int id : userIds) {
            User oxUser = userService.getUser(id, ownerKey.getContextid());
            if (oxUser == null) {
                throw GuardSharingExceptionCodes.UNABLE_TO_FIND_OX_USER.create();  // Really shouldn't happen
            }
            String email = oxUser.getMail();
            GuardKeys userKey = keyService.getKeys(email);
            if (userKey != null) {
                keys.add(new RecipKey(userKey).getEncryptionKey());
            } else {
                OXUser userCheck = Services.getService(OXUserService.class).getUser(email);  // Let's check if OX user
                RecipKey rKey = new RecipKey();
                if (userCheck != null && !userCheck.isGuest()) {
                    rKey.setCid(userCheck.getContextId());
                    rKey.setUserid(userCheck.getId());
                    rKey.setLang(userCheck.getLanguage());
                    rKey.setName(userCheck.getName());
                    rKey.setEmail(userCheck.getEmail());
                    rKey.setPgp(true);
                } else { // OK, it's going to be a guest user
                    rKey.setCid(ownerKey.getContextid() * -1);
                    rKey.setLang(ownerKey.getLanguage());
                    rKey.setName(oxUser.getDisplayName());
                    rKey.setEmail(email);
                    rKey.setPgp(true);
                    rKey.setGuest(true);
                }
                RecipKey newKey = Services.getService(AccountCreationService.class).createUserFor(rKey);
                if (newKey == null) {
                    throw GuardSharingExceptionCodes.UNABLE_TO_GET_KEY.create(email);
                }
                keys.add(newKey.getEncryptionKey());
            }
        }
        return keys;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.sharing.SharingService#getAdditionalShareRecipients(java.util.List, java.util.List)
     */
    @Override
    public List<String> getAdditionalShareRecipients(List<String> recipients, List<String> emails) throws OXException {
        ArrayList<String> allRecips = new ArrayList<String> ();
        allRecips.addAll(recipients);
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        for (String email: emails) {
            GuardKeys key = keyService.getKeys(email);
            if (key == null) {
                throw GuardSharingExceptionCodes.UNABLE_TO_GET_KEY.create(email);
            }
            String id = key.getContextid() + "/" + key.getUserid();
            if (!allRecips.contains(id)) {  // Avoid duplicates
                allRecips.add(id);
            }

        }
        return allRecips;
    }


}
