/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.ratifier.internal;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.idn.IDNA;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.oxapi.Normalizer;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.ratifier.exceptions.GuardRatifierExceptionCodes;

/**
 * {@link GuardRatifierServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardRatifierServiceImpl implements GuardRatifierService {

    private static final Logger logger = LoggerFactory.getLogger(GuardRatifierServiceImpl.class);
    private static final String ip4Regex = "\\b([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})\\b";
    private static final String ip6Regex = "\\A(?:" +               //Anchor address
        "(?:" +                                                     // Mixed
        "(?:[A-F0-9]{1,4}:){6}" +                                   // Non-compressed
        "|(?=(?:[A-F0-9]{0,4}:){2,6}" +                             // Compressed with 2 to 6 colons
        "(?:[0-9]{1,3}\\.){3}[0-9]{1,3}" +                          //    and 4 bytes
        "\\z)" +                                                    //    and anchored
        "(([0-9A-F]{1,4}:){1,5}|:)((:[0-9A-F]{1,4}){1,5}:|:)" +     //    and at most 1 double colon
        "|::(?:[A-F0-9]{1,4}:){5}" +                                // Compressed with 7 colons and 5 numbers
        ")" +
        "(?:(?:25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])\\.){3}" +  // 255.255.255.
        "(?:25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])" +          // 255
        "|" +                                                       // Standard\n" +
        "(?:[A-F0-9]{1,4}:){7}[A-F0-9]{1,4}" +                      // Standard
        "|" +                                                       // Compressed\n" +
        "(?=(?:[A-F0-9]{0,4}:){0,7}[A-F0-9]{0,4}" +                 // Compressed with at most 7 colons
        "\\z)" +                                                    //    and anchored
        "(([0-9A-F]{1,4}:){1,7}|:)((:[0-9A-F]{1,4}){1,7}|:)" +      //    and at most 1 double colon
        "|(?:[A-F0-9]{1,4}:){7}:|:(:[A-F0-9]{1,4}){7}" +            // Compressed with 8 colons
        ")\\z";                                                     // Anchor address

    /**
     * Initialises a new {@link GuardRatifierServiceImpl}.
     */
    public GuardRatifierServiceImpl() {
        super();
    }

    @Override
    public boolean isValid(String email) {
        if (Strings.isEmpty(email)) {
            return false;
        }
        if (!email.contains("@")) {
            return false;
        }
        try {
            String aceConformMailAddress = IDNA.toACE(new String(email));
            InternetAddress e = new InternetAddress(aceConformMailAddress);
            e.validate();
        } catch (AddressException e) {
            logger.warn("{}", e);
            return false;
        }
        return true;
    }

    @Override
    public void validate(String email) throws OXException {
        if (!isValid(email)) {
            throw GuardRatifierExceptionCodes.INVALID_EMAIL_ADDRESS.create(email);
        }
    }

    @Override
    public String normalise(String data) throws OXException {
        return new Normalizer().normalizeString(data);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.ratifier.GuardRatifierService#verifyRemoteIP(java.lang.String)
     */
    @Override
    public String verifyRemoteIP(String ipAddress) {
        return verifyIP(ipAddress, false);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.ratifier.GuardRatifierService#verifyIPAddress(java.lang.String)
     */
    @Override
    public String verifyIPAddress(String ipAddress) {
        return verifyIP(ipAddress, true);
    }

    private String verifyIP(String ipAddress, boolean allowLocal) {
        if (ipAddress == null) return null;
        if (ipAddress.matches(ip4Regex)) {
            // ip4 address
            if (allowLocal) return ipAddress;
            // Check if local IP
            Pattern p = Pattern.compile(ip4Regex);
            Matcher m = p.matcher(ipAddress);
            if (!m.find()) return null;
            String first = m.group(1);
            String second = m.group(2);
            int secondInt = 0;
            if (second == null) {
                return null;
            }
            secondInt = Integer.parseInt(second);
            if (ipAddress.equals("127.0.0.1")) return null;  // Localhost
            // Check IP4 local addresses
            if (first.equals("10") ||
                (first.equals("192") && second.equals("168")) ||
                (first.equals("172") && (secondInt >= 16) && (secondInt <= 31))) {
                return null;
            }
            return ipAddress;
        }
        ipAddress = ipAddress.toUpperCase();
        if (ipAddress.matches(ip6Regex)) {
            if (allowLocal) return ipAddress;
            // Check if local address
            if (ipAddress.equals("0:0:0:0:0:0:0:1") || ipAddress.equals("::1")) return null;  // Localhost
            if (ipAddress.startsWith("FD")) return null;  // IP6 local address
            return (ipAddress);
        }
        return null;
    }
}
