/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.oxapi;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import javax.servlet.http.Cookie;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.oxapi.exceptions.OXApiExceptionCodes;

/**
 * Handles getting OX Authorization Cookie, parsing, and returning info
 *
 * @author greg
 */
final class cookie {

    private final String name;
    private final String value;

    public cookie(String name, String value) {
        this.name = name;
        this.value = value;
    }

    public String getName() {
        return name;
    }

    public String getValue() {
        return value;
    }
}

public class OxCookie {

    private static final Logger LOG = LoggerFactory.getLogger(OxCookie.class);

    private String JSESSIONID = "";

    private final ArrayList<cookie> secretcookie = new ArrayList<cookie>();

    private final ArrayList<cookie> publicsession = new ArrayList<cookie>();

    private final ArrayList<cookie> session = new ArrayList<cookie>();

    private String oxReaderID = "";

    private String language = "en_US";

    private Boolean cookieAuth = false;

    public OxCookie(Cookie[] cookies) {
        try {
            if (cookies == null) {
                return;
            }
            if (cookies.length > 0) {// Make sure the cookies are present
                for (int i = 0; i < cookies.length; i++) {
                    String name = cookies[i].getName();
                    if (name.startsWith("JSESSIONID")) {
                        setJSESSIONID(Strings.removeCarriageReturn(cookies[i].getValue()));
                    }
                    if (name.startsWith("open-xchange-secret")) {
                        cookie ck = new cookie(name, Strings.removeCarriageReturn(cookies[i].getValue()));
                        secretcookie.add(ck);
                        cookieAuth = true; // Set as authorized if we have the secret data
                    }
                    if (name.startsWith("language")) {
                        language = Strings.removeCarriageReturn(cookies[i].getValue());
                    }
                    if (name.startsWith("open-xchange-session")) {
                        cookie ck = new cookie(name, Strings.removeCarriageReturn(cookies[i].getValue()));
                        session.add(ck);
                        cookieAuth = true; // Set as authorized if we have the secret data
                    }
                    if (name.startsWith("open-xchange-public-session")) {
                        cookie ck = new cookie(name, Strings.removeCarriageReturn(cookies[i].getValue()));
                        publicsession.add(ck);
                    }
                    if (name.startsWith("OxReaderID")) {
                        setOxReaderID(Strings.removeCarriageReturn(cookies[i].getValue()));
                    }
                }
            }
        } catch (Exception ex) {
            LOG.error("Error handling cookie", ex);
        }
    }

    public String getJSESSIONID() {
        return JSESSIONID;
    }

    public Boolean isAuth() {
        return cookieAuth;
    }

    /**
     * Generate cookie header
     *
     * @return
     */
    public String getCookieHeader() {
        StringBuilder sb = new StringBuilder();
        sb.append("language=" + language + "; JSESSIONID=" + getJSESSIONID() + "; ");
        for (int i = 0; i < session.size(); i++) {
            sb.append(session.get(i).getName() + "=" + session.get(i).getValue() + "; ");
        }
        for (int j = 0; j < secretcookie.size(); j++) {
            sb.append(secretcookie.get(j).getName() + "=" + secretcookie.get(j).getValue() + "; ");
        }
        for (int k = 0; k < publicsession.size(); k++) {
            sb.append(publicsession.get(k).getName() + "=" + publicsession.get(k).getValue() + "; ");
        }
        return sb.toString();
    }

    /**
     * Create a JSON Object of the cookie
     *
     * @return
     */
    public JsonObject getCookieJson() {
        JsonObject cookie = new JsonObject();
        cookie.addProperty("JSESSIONID", getJSESSIONID());
        for (int i = 0; i < session.size(); i++) {
            cookie.addProperty(session.get(i).getName(), session.get(i).getValue());
        }
        for (int j = 0; j < secretcookie.size(); j++) {
            cookie.addProperty(secretcookie.get(j).getName(), secretcookie.get(j).getValue());
        }
        for (int k = 0; k < publicsession.size(); k++) {
            cookie.addProperty(publicsession.get(k).getName(), publicsession.get(k).getValue());
        }
        return cookie;
    }

    public void setJSESSIONID(String jSESSIONID) {
        JSESSIONID = jSESSIONID;
    }

    public String getOxReaderID() {
        return oxReaderID;
    }

    public void setOxReaderID(String oxReaderID) {
        this.oxReaderID = oxReaderID;
    }

    public String getSessionID () {
        if(this.session.size() > 0) {
            return session.get(0).getValue();
        }
        return null;
    }
}
