/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.MimeMessage;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.lang3.ArrayUtils;
import org.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guest.GuardGuestService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEMailMetadataService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEmailMetadata;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.mime.encryptor.Decryptor;
import com.openexchange.guard.mime.encryptor.Encryptor;
import com.openexchange.guard.mime.encryptor.ProcessMime;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.osgi.Services;
import com.openexchange.guard.mime.encryptor.pgp.guest.PGPGuestEncryptor;
import com.openexchange.guard.mime.encryptor.pgp.inline.PGPInlineDecryptor;
import com.openexchange.guard.mime.encryptor.pgp.inline.PGPInlineEncryptor;
import com.openexchange.guard.mime.encryptor.pgp.pgpmime.GuardPGPMimeEncryptor;
import com.openexchange.guard.mime.encryptor.pgp.pgpmime.PGPMimeDecryptor;
import com.openexchange.guard.mime.helpfiles.HelpFileService;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.mime.services.MimeParser;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.pgpcore.services.PGPMimeCryptoService;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.GuardCapabilities.Permissions;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.mail.PGPMimeService;

/**
 * {@link PGPProcessMime}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class PGPProcessMime implements ProcessMime {

    private static Logger LOG = LoggerFactory.getLogger(PGPProcessMime.class);

    private final PGPCryptoService pgpCryptoService;
    private final PGPMimeCryptoService pgpMimeCryptoService;
    private final GuardConfigurationService guardConfigurationService;
    private final GuardGuestEMailMetadataService guardMetaDataService;
    private final EmailStorage emailStorage;

    private HelpFileService helpFileService;

    private PGPMimeService pgpMimeService;

    private GuardNotificationService guardNotificationService;

    private GuardGuestService guestService;

    private MailCreatorService mailCreatorService;

    /**
     * Initializes a new {@link PGPProcessMime}.
     *
     * @param pgpCryptoService The {@link PGPCryptoService} to use for mime encryption
     */
    public PGPProcessMime(GuardConfigurationService guardConfigurationService,
                          PGPCryptoService pgpCryptoService,
                          PGPMimeService pgpMimeService,
                          PGPMimeCryptoService pgpMimeCryptoService,
                          HelpFileService helpFileService,
                          GuardGuestService guestService,
                          GuardNotificationService guardNotificationService,
                          GuardGuestEMailMetadataService guardMetaDataService,
                          EmailStorage emailStorage,
                          MailCreatorService mailCreatorService) {
        this.pgpCryptoService = pgpCryptoService;
        this.pgpMimeService = pgpMimeService;
        this.pgpMimeCryptoService = pgpMimeCryptoService;
        this.guardConfigurationService = guardConfigurationService;
        this.helpFileService = helpFileService;
        this.guestService = guestService;
        this.guardNotificationService = guardNotificationService;
        this.guardMetaDataService = guardMetaDataService;
        this.emailStorage = emailStorage;
        this.mailCreatorService = mailCreatorService;

    }

    @Override
    public String doEncrypt(GuardParsedMimeMessage msg, int userId, int contextId, String password) throws OXException {

        //Assures appropriated Guard permissions for user
        // If doesn't have guard-mail, possible secure reply.  Verify all users have keys
        // No new users allowed.
        // TODO enhance this
        if (!checkMailPermission(userId, contextId) && !msg.isDraft()) {
            try {
                // Check recipients as draft email (no new keys created)
                RecipientParser testRecipParse = new RecipientParser(msg, userId, contextId, true);
                ArrayList<RecipKey> checkRecipts = testRecipParse.getAllRecipients();
                // Verify found keys same length as number recipients
                if (checkRecipts.size() != msg.getRecipients().length()) {
                    LOG.error("Attempt to send email from guard only capable account, with new recipients");
                    throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
                }
            } catch (AddressException | JSONException e) {
               LOG.error("Error checking for added recipients for guard capability only send", e);
               throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create(e);
            }
        }

        try {
            String origMessageID = msg.getMessage().getMessageID();
            RecipientParser recipParse = new RecipientParser(msg, userId, contextId, msg.isDraft());
            ArrayList<RecipKey> recips = recipParse.getPgpRecipients();
            ArrayList<RecipKey> guests = recipParse.getGuestRecipients();
            ArrayList<RecipKey> allRecipts = recipParse.getAllRecipients();
            // Handle Guests
            if (guests.size() > 0 && !msg.isDraft()) {  // Won't create guests for draft messages
                Encryptor guestEncryptor = new PGPGuestEncryptor(pgpCryptoService, pgpMimeService, guestService, helpFileService, msg, guests, userId, contextId, password);
                if (!guestEncryptor.doEncryptAndSend()) {
                    throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create("Unable to send Guest emails");
                }
            }
            // Verify sender is Guest guest user, and change userid/cid to guard user
            if (msg.senderIsGuest()) {
                Email emailResult = emailStorage.getByEmail(msg.getFromAddress().getAddress());
                if (emailResult == null) {
                    throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create();
                }
                userId = emailResult.getUserId();  // Need to change the UserId/Cid from the OX Guest to Guard userid/cid
                contextId = emailResult.getContextId();
                if(!msg.isDraft()) {
                    checkGuestSendOK(userId, contextId, msg.getGuardMessagRef(), recips);
                }
            }


            Encryptor encryptor = null;
            if (recipParse.isInline() || msg.isInline()) {
                encryptor = new PGPInlineEncryptor(pgpCryptoService, msg, recips, userId, contextId, password);
            } else {
                if (recips.size() > 0) {
                    encryptor = new GuardPGPMimeEncryptor(pgpCryptoService, helpFileService, msg, recips, userId, contextId, password);
                } else {
                    if (guests.size() > 0) {  // No recips, but we had guests, we need to send this for the sent folder
                        encryptor = new GuardPGPMimeEncryptor(pgpCryptoService, helpFileService, msg, guests, userId, contextId, password);
                    }
                }
            }
            if(encryptor != null) {
                //encrypt the message
                final MimeMessage  encryptedResult = encryptor.doEncrypt();

                if(msg.senderIsGuest() && !msg.isDraft()) {
                    //Sending on behalf of a guest
                    send(encryptedResult, msg.getSenderEmail(), contextId, recips, msg.getSenderIP());

                    //Remove the all recipients handled by Guard
                    return convertToString(removeRecipients(encryptedResult, allRecipts, origMessageID));
                }

                //Just remove the guests handled by Guard
                return convertToString(removeRecipients(encryptedResult, guests, origMessageID));
            }
            else {
                throw MimeEncryptorExceptionCodes.NO_RECIPIENTS.create();
            }
        } catch (JSONException | MessagingException | IOException e) {
            LOG.error("Error encrypting email ", e);
            throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create(e.getMessage());
        }
    }

    private void send(Message message, String senderEmail, int senderCid, ArrayList<RecipKey> recipients, String remoteIp) throws OXException {
        String fromAddress = getFromEmailAddress(recipients, senderEmail);
        for(RecipKey recipient : recipients) {
            this.guardNotificationService.sendMessage(message, fromAddress, senderCid, recipient.getLang(), remoteIp);
        }
    }

    private String getFromEmailAddress(List<RecipKey> recipients, String fromaddres) throws OXException {
        int cid = 0;
        int id = 0;
        for (int i = 0; i < recipients.size(); i++) {// Find at least one member with positive cid, and use that configured address
            if (recipients.get(i).getCid() > 0) {
                cid = recipients.get(i).getCid();
                id = recipients.get(i).getUserid();
                break;
            }
        }
        return this.mailCreatorService.getFromAddress(fromaddres, fromaddres, id, cid).get(1);
    }

    /**
     * Internal method to make sure Guest not replying to new recipients.  Maintaining email chain
     * @param userid  Guard Guest userid
     * @param contextId Guard guest contextId
     * @param msgRef  Original message ref
     * @param recipients  List of recipients
     * @throws OXException
     */
    private void checkGuestSendOK(int userid, int contextId, String msgRef, ArrayList<RecipKey> recipients) throws OXException {
        if (msgRef != null) {
            GuardGuestEmailMetadata meta = guardMetaDataService.get(contextId, userid, msgRef);
            if (meta == null) {
                throw MimeEncryptorExceptionCodes.RECIPIENTS_NOT_IN_ORIGINAL_CHAIN.create("Unable to retrieve original meta data");
            }
            String[] metaTos = ArrayUtils.addAll(
                meta.getTo(),
                ArrayUtils.addAll(meta.getCc(), meta.getFrom()));

            for (RecipKey recip : recipients) {
                String email = recip.getEmail();
                boolean found = false;
                for (String inChain : metaTos) {
                    if (inChain.toLowerCase().contains(email.toLowerCase())) {
                        found = true;
                    }
                }
                if (!found) {
                    throw MimeEncryptorExceptionCodes.RECIPIENTS_NOT_IN_ORIGINAL_CHAIN.create(email);
                }
            }
        }
        else {
            throw MimeEncryptorExceptionCodes.RECIPIENTS_NOT_IN_ORIGINAL_CHAIN.create("");
        }
    }

    /**
     * @param userId
     * @param contextId
     * @throws OXException
     */
    private void requirePermission(int userId, int contextId, boolean sending) throws OXException {
        // Check user permission
        GuardCapabilities guardUserCapabilities = Services.getService(OXUserService.class).getGuardCapabilieties(contextId, userId);
        if (sending && !guardUserCapabilities.hasPermission(Permissions.MAIL)) {
            // TODO
            // Handle replies to encrypted emails when the user doesn't have Guard enabled
            // Guest replies when moved to core?
            throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
        } else {
            if (!guardUserCapabilities.hasMinimumPermission()) {
                throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
            }
        }
    }

    /**
     * Checks to see if has Guard-Mail capability.  If so, returns true.  Of Guard only, returns false
     * If no Guard permissions, throws Not_Authorized
     * @param userId
     * @param contextId
     * @return
     * @throws OXException
     */
    private boolean checkMailPermission (int userId, int contextId) throws OXException {
        GuardCapabilities guardUserCapabilities = Services.getService(OXUserService.class).getGuardCapabilieties(contextId, userId);
        if (guardUserCapabilities.hasPermission(Permissions.MAIL)) {
            return true;
        }
        if (guardUserCapabilities.hasMinimumPermission()) {
            return false;
        }
        throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.mime.encryptor.ProcessMime#doDecrypt(com.openexchange.guard.mime.services.GuardParsedMimeMessage, java.lang.String)
     */
    @Override
    public MimeMessage doDecrypt(MimeMessage message, int userId, int contextId, UserIdentity userIdentity) throws OXException {

        //Assures appropriated Guard permissions for user
        requirePermission(userId, contextId, false);

        try {
            MimeParser mimeParser = new MimeParser(message);
            Decryptor decryptor = null;

            if (mimeParser.isPGPMime()) {
                decryptor = new PGPMimeDecryptor(pgpMimeCryptoService, guardConfigurationService, userIdentity);
            } else if (mimeParser.isPGPInline()) {
                decryptor = new PGPInlineDecryptor(pgpCryptoService, guardConfigurationService, userIdentity);
            }

            if (decryptor != null) {
                return decryptor.doDecrypt(message);
            }
            return message;
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    private String convertToString(MimeMessage message) throws IOException, MessagingException {
        ByteArrayOutputStream wrapped_out = new ByteArrayOutputStream();
        message.writeTo(wrapped_out);
        byte[] wrapped_bytes = wrapped_out.toByteArray();
        return new String(wrapped_bytes, "UTF-8");
    }

    /**
     * Add header to the email to remove the Guests from the list of recipients that the middleware will send to
     *
     * @param message
     * @param recipients
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    private MimeMessage removeRecipients(MimeMessage message, ArrayList<RecipKey> recipients, String origMessageID) throws MessagingException, IOException {
        if (message == null) {
            return null;
        }
        for (RecipKey key : recipients) {
            message.addHeader("Remove-recip", key.getEmail());
        }
        message.removeHeader("X-OxGuard-PIN");  // Remove record of pin from non-encrypted email
        message.saveChanges();
        if (origMessageID != null) {  // Save changes and enryption may change messageID.  Restore original here
            message.setHeader("Message-ID", origMessageID);
        }
        return message;
    }
}
